<?php

declare(strict_types=1);

namespace AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Exception\BadResponseException;
use AlloCine\GraphClient\Bundle\Exception\FileNotFoundException;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use stdClass;

class GraphClient
{
    private ?string $query = null;

    private ?string $queryName = null;

    private ?string $preparedQuery;

    private array $variables = [];

    private ?string $uri = null;

    public function __construct(
        protected GraphApiBridge $bridge,
        private readonly GraphCursor $cursor,
        private readonly QueryParser $parser,
        private readonly PhpArrayAdapter $warmUp
    ) {
    }

    protected function computeHash(): string
    {
        return \md5(\sprintf('%s_%s', $this->queryName, \serialize($this->variables)));
    }

    /**
     * Compute hash for testing purpose (building and finding fixtures)
     */
    protected function computeTestHash(): string
    {
        $variables = $this->variables;
        \array_walk_recursive(
            $variables,
            static function (&$value): void {
                if (\is_string($value) && 0 < \preg_match('#^(\d{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2]\d|3[0-1])$#', $value)) {
                    $value = null;
                }
            }
        );

        $query = \preg_replace('#(\d{4}-)?(0[1-9]|1[0-2])-(0[1-9]|[1-2]\d|3[0-1])#', '', $this->getQuery());

        return \md5(\sprintf('%s_%s_%s', $this->uri, $query, \serialize($variables)));
    }

    public function getQuery(): string
    {
        return $this->query ?? $this->preparedQuery;
    }

    public function setUri(string $uri): GraphClient
    {
        $this->uri = $uri;

        return $this;
    }

    public function setVariables(array $variables): GraphClient
    {
        $this->variables = $variables;

        return $this;
    }

    public function getAfterCursor(int $cursor, int $offset = 0): ?string
    {
        return $this->cursor->getAfterCursor($cursor, $offset);
    }

    /**
     * @throws InvalidArgumentException
     */
    public function prepare(string $query): GraphClient
    {
        $this->preparedQuery = $this->parser
            ->setFragments(
                $this->warmUp
                    ->getItem(GraphApiCacheWarmer::CACHED_FRAGMENT_KEY)
                    ->get()
            )
            ->parseQuery($query);

        return $this;
    }

    /**
     * @throws InvalidArgumentException
     */
    public function generateQuery(?string $queryName = null): GraphClient
    {
        $this->queryName = $queryName;
        $query = $queryName ? $this->warmUp->getItem($queryName)->get() : null;
        if (null === $this->preparedQuery && null === $query) {
            throw new FileNotFoundException("The graph query file $queryName does not exist");
        }

        if (null !== $this->preparedQuery) {
            $query = $this->preparedQuery;
        }
        $this->parseQuery($query);

        return $this;
    }

    /**
     * @throws BadResponseException
     */
    public function getResults(): ?stdClass
    {
        $result = $this->bridge->query($this->query, $this->computeTestHash());
        $this->init();

        return $result;
    }

    private function parseQuery(string $query): void
    {
        $variables = $this->variables === [] ? '' : sprintf(', "variables": %s',
            \json_encode($this->variables, JSON_THROW_ON_ERROR)
        );

        $this->query = \sprintf(
            '{"query": "%s"%s}',
            addslashes((string) preg_replace('@\s+@', ' ', $query)),
            $variables
        );
    }

    private function init(): void
    {
        $this->query = null;
        $this->preparedQuery = null;
        $this->uri = null;
        $this->variables = [];
        $this->queryName = null;
    }
}
