<?php

namespace Tests\AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Client\GraphApiBridge;
use AlloCine\GraphClient\Bundle\Client\GraphClient;
use AlloCine\GraphClient\Bundle\Client\GraphCursor;
use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClientTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var PhpArrayAdapter
     */
    private $cacheAdapter;

    /**
     * @var FilesystemAdapter
     */
    private $fileSystemAdapter;

    /**
     * @throws \Exception|\FileNotFoundException
     */
    public function setUp()
    {
        $this->setCacheFragments();

        $this->bridge = $this->getMockBuilder(GraphApiBridge::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->cursor = new GraphCursor();
    }

    public function testCursorValues()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $this->assertNull($service->getAfterCursor(1, 0));
        $this->assertSame(base64_encode('arrayconnection:9'), $service->getAfterCursor(2, 10));
        $this->assertSame(base64_encode('arrayconnection:29'), $service->getAfterCursor(2, 30));
    }

    /**
     * @expectedException \FileNotFoundException
     * @expectedExceptionMessage The graph fragment errorFragment does not exist
     * @throws \Exception|\FileNotFoundException
     */
    public function testPrepareError()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testWithError {
                ...errorFragment
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws \Exception|\FileNotFoundException
     */
    public function testPrepareWithoutFragment()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query);
        $this->assertSame($query, $service->getQuery());
    }

    /**
     * @throws \Exception|\FileNotFoundException
     */
    public function testPrepareWithFragment()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        }
GRAPHQL;
        $expected =
            <<<GRAPHQL
        {
            testReal {
                ...realFragment
            }
        } fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL;
        $service->prepare($query);
        var_dump(rtrim($expected), rtrim($service->getQuery()));
        $this->assertSame(rtrim($expected), rtrim($service->getQuery()));
    }

    /**
     * @throws \Exception|\FileNotFoundException
     */
    public function testGenerateQuery()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }"}', $service->getQuery());
    }

    /**
     * @throws \Exception|\FileNotFoundException
     */
    public function testGenerateQueryWithVariables()
    {
        $service = new GraphClient($this->bridge, $this->cursor, new QueryParser(), $this->cacheAdapter);

        $query =
            <<<GRAPHQL
        {
            testReal {
                test
            }
        }
GRAPHQL;
        $service->prepare($query)->setVariables(['order' => '"ORDER"'])->generateQuery();
        $this->assertSame('{"query": " { testReal { test } }", "variables": {"order": "ORDER"}}', $service->getQuery());
    }

    /**
     * @throws \Exception|\FileNotFoundException
     */
    private function setCacheFragments()
    {
        $cacheDir = __DIR__ . '/../Resources/cache/queries.php';
        $this->fileSystemAdapter = new FilesystemAdapter('', 0, __DIR__ . '/../Resources/cache/filesystem');
        $this->cacheAdapter = new PhpArrayAdapter($cacheDir, $this->fileSystemAdapter);
        $sources = ['queries' => [__DIR__ . '/../Resources/graph/queries'], 'fragments' => [__DIR__ . '/../Resources/graph/fragments']];
        $service = new GraphApiCacheWarmer($this->cacheAdapter, new QueryParser(), $sources, '.graphql');
        $service->warmUp('');
    }
}
