<?php

namespace AlloCine\GraphClient\Bundle\Cache;

use AlloCine\GraphClient\Bundle\Parser\QueryParser;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerInterface;

class GraphApiCacheWarmer implements CacheWarmerInterface
{
    const CACHED_FRAGMENT_KEY = 'cachedFragments';

    /**
     * @var PhpArrayAdapter
     */
    private $cache;

    /**
     * @var QueryParser
     */
    private $parser;

    /**
     * @var array
     */
    private $sources;

    /**
     * @var string
     */
    private $extension;

    /**
     * @var array
     */
    private $queries = [];

    /**
     * @var array
     */
    private $fragments = [];

    /**
     * @param PhpArrayAdapter $cache
     * @param QueryParser     $parser
     * @param array           $sources ['queries' => array, 'fragments' => array]
     * @param string          $extension
     */
    public function __construct(
        PhpArrayAdapter $cache,
        QueryParser $parser,
        array $sources,
        string $extension
    ) {
        $this->cache = $cache;
        $this->parser = $parser;
        $this->sources = $sources;
        $this->extension = $extension;
        $this->init();
    }

    private function init()
    {
        $this->storedQueries('queries');
        $this->storedQueries('fragments');
    }

    /**
     * @param string $arrayName
     */
    private function storedQueries(string $arrayName)
    {
        $finder = new Finder();
        $templates = $finder->files()->in($this->sources[$arrayName]);
        $extension = str_replace('.', '', $this->extension);
        /** @var SplFileInfo $template */
        foreach ($templates as $template) {
            if ($template->getExtension() === $extension) {
                $fragmentName = str_replace($this->extension, '', $template->getFilename());
                $this->$arrayName[$fragmentName] = $template->getContents();
            }
        }
    }

    /**
     * @param string $cacheDir
     *
     * @throws \FileNotFoundException|\Exception
     */
    public function warmUp($cacheDir)
    {
        $this->parser->setFragments($this->fragments);
        $queries = [];

        foreach ($this->queries as $name => $query) {
            $queries[$name] = $this->parser->parseQuery($query);
        }
        $queries[self::CACHED_FRAGMENT_KEY] = $this->fragments;

        $this->cache->warmUp($queries);
    }

    /**
     * {@inheritsDoc}
     */
    public function isOptional()
    {
        return false;
    }
}
