<?php

namespace Tests\AlloCine\GraphClient\Bundle\Client;

use AlloCine\GraphClient\Bundle\Cache\GraphApiCacheWarmer;
use AlloCine\GraphClient\Bundle\Client\GraphApiBridge;
use AlloCine\GraphClient\Bundle\Client\GraphClient;
use AlloCine\GraphClient\Bundle\Client\GraphCursor;
use Symfony\Component\Cache\Adapter\FilesystemAdapter;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class GraphClientTest extends \PHPUnit_Framework_TestCase
{
    /**
     * @var GraphApiBridge
     */
    private $bridge;

    /**
     * @var GraphCursor
     */
    private $cursor;

    /**
     * @var PhpArrayAdapter
     */
    private $cacheAdapter;

    /**
     * @var FilesystemAdapter
     */
    private $fileSystemAdapter;

    public function setUp()
    {
        $this->setCacheFragments();

        $this->bridge = $this->getMockBuilder(GraphApiBridge::class)
            ->disableOriginalConstructor()
            ->getMock();

        $this->cursor = new GraphCursor();
    }

    /**
     * @expectedException \FileNotFoundException
     * @expectedExceptionMessage The graph fragment errorFragment does not exist
     */
    public function testClientWithoutFragment()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $service->prepare(
            <<<GRAPHQL
        {
            testWithError {
                ...errorFragment
            }
        }
GRAPHQL
        )->generateQuery('testError');
    }

    /**
     * @expectedException \Exception
     * @expectedExceptionMessage A criterion in graph query wasn't replaced
     */
    public function testClientWithoutCriterion()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $service->prepare(
            <<<GRAPHQL
        {
            testWithError {
                ...withoutCriterion
            }
        }
GRAPHQL
        )->generateQuery('testError');
    }

    /**
     * @expectedException \Exception
     * @expectedExceptionMessage The criterion values must be an array
     */
    public function testClientWithFalseCriterion()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $service->prepare(
            <<<GRAPHQL
        {
            testWithError {
                id
            }
        }
GRAPHQL
        )->generateQuery('testError', ['withoutCriterion' => false]);
    }

    public function testQueryConstructionWithoutCriterion()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $query = $service->generateQuery('testReal')->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testReal{
        id
    }
}    
GRAPHQL
        ),
            rtrim($query)
        );
    }

    public function testQueryConstructionWithCriterion()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $query = $service->generateQuery('testRealWithCriterion', ['criterions' => ['test' => 10]])->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testReal (test: 10){
        id
    }
}    
GRAPHQL
        ),
            rtrim($query)
        );
    }

    public function testQueryConstructionWithCriterionAndFragments()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $query = $service->generateQuery('testRealWithCriterionAndFragments', ['criterions' => ['test' => 10]])->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testReal (test: 10){
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($query)
        );
    }

    public function testQueryConstructionWithTwoFragments()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $query = $service->generateQuery('testRealWithTwoFragments')->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testTwoReal (test: 10){
        ...realFragment
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($query)
        );
    }

    public function testQueryConstructionWithTwoFragmentsInTwoQueries()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);
        $query = $service->generateQuery('testRealWithTwoFragments')->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testTwoReal (test: 10){
        ...realFragment
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($query)
        );

        $query = $service->generateQuery('testRealWithTwoFragments2')->getQuery();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testTwoReal2 (test: 10){
        ...realFragment
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($query)
        );
    }

    public function testCursorValues()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);

        $this->assertNull($service->getAfterCursor(1, 0));
        $this->assertSame(base64_encode('arrayconnection:9'), $service->getAfterCursor(2, 10));
        $this->assertSame(base64_encode('arrayconnection:29'), $service->getAfterCursor(2, 30));
    }

    private function setCacheFragments()
    {
        $cacheDir = __DIR__ . '/../Resources/cache/queries.php';
        $this->fileSystemAdapter = new FilesystemAdapter('', 0, __DIR__ . '/../Resources/cache/filesystem');
        $this->cacheAdapter = new PhpArrayAdapter($cacheDir, $this->fileSystemAdapter);
        $service = new GraphApiCacheWarmer($this->cacheAdapter, [__DIR__ . '/../Resources/graph/'], '.graphql');
        $service->warmUp('');
    }

    public function testStoredAsyncQuery()
    {
        $service = new GraphClient($this->bridge, $this->cursor, $this->fileSystemAdapter, $this->cacheAdapter);

        $queries = $service
                ->setHash('123456')
                ->storedAsyncQuery('testRealWithTwoFragments2')
                ->setHash('7891011')
                ->storedAsyncQuery('testRealWithTwoFragments')
                ->getAsyncQueries();

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testTwoReal (test: 10){
        ...realFragment
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($queries['testRealWithTwoFragments']['body'])
        );
        $this->assertSame('7891011', $queries['testRealWithTwoFragments']['hash']);

        $this->assertSame(rtrim(
            <<<GRAPHQL
{
    testTwoReal2 (test: 10){
        ...realFragment
        ...realFragment
    }
}
fragment realFragment on real {
    test {
        id
    }
}
GRAPHQL
        ),
            rtrim($queries['testRealWithTwoFragments2']['body'])
        );
        $this->assertSame('123456', $queries['testRealWithTwoFragments2']['hash']);
    }
}
