<?php

namespace AlloCine\DoctrineCacheRediscluster\Cache\Doctrine;

use Doctrine\Common\Cache\Cache;
use Doctrine\Common\Cache\CacheProvider;

class CacheRedisClusterAdapter extends CacheProvider
{
    /**
     * @var \RedisCluster|null
     */
    private $redis;

    /**
     * Sets the redis instance to use.
     *
     * @param \RedisCluster $redis
     */
    public function setRedis($redis)
    {
        $redis->setOption(
            \RedisCluster::OPT_SERIALIZER,
            $this->getSerializerValue()
        );
        $this->redis = $redis;
    }

    /**
     * Gets the redis instance used by the cache.
     *
     * @return \RedisCluster|null
     */
    public function getRedis()
    {
        return $this->redis;
    }

    /**
     * {@inheritdoc}
     */
    protected function doFetch($id)
    {
        return $this->redis->get($id);
    }

    /**
     * {@inheritdoc}
     */
    protected function doFetchMultiple(array $keys)
    {
        $fetchedItems = array_combine($keys, $this->redis->mget($keys));

        //Redis mget returns false for keys that do not exist. So we need to filter those out unless it's the real data.
        $foundItems = [];

        foreach ($fetchedItems as $key => $value) {
            if (false !== $value || $this->redis->exists($key)) {
                $foundItems[$key] = $value;
            }
        }

        return $foundItems;
    }

    /**
     * {@inheritdoc}
     */
    protected function doSaveMultiple(array $keysAndValues, $lifetime = 0)
    {
        if ($lifetime) {
            $success = true;

            // Keys have lifetime, use SETEX for each of them
            foreach ($keysAndValues as $key => $value) {
                if (!$this->redis->setex($key, $lifetime, $value)) {
                    $success = false;
                }
            }

            return $success;
        }

        // No lifetime, use MSET
        return (bool) $this->redis->mset($keysAndValues);
    }

    /**
     * {@inheritdoc}
     */
    protected function doContains($id)
    {
        return $this->redis->exists($id);
    }

    /**
     * {@inheritdoc}
     */
    protected function doSave($id, $data, $lifeTime = 0)
    {
        if ($lifeTime > 0) {
            return $this->redis->setex($id, $lifeTime, $data);
        }

        return $this->redis->set($id, $data);
    }

    /**
     * {@inheritdoc}
     */
    protected function doDelete($id)
    {
        return $this->redis->del($id) >= 0;
    }

    /**
     * {@inheritdoc}
     */
    protected function doDeleteMultiple(array $keys)
    {
        return call_user_func_array([$this->redis, "del"], $keys) >= 0;
    }

    /**
     * {@inheritdoc}
     */
    protected function doFlush()
    {
        foreach ($this->redis->seeds as $seed) {
            $this->redis->flushDB(str_replace(':', ',', $seed));
        }

        return true;
    }

    /**
     * {@inheritdoc}
     */
    protected function doGetStats()
    {
        $info = $this->redis->info();

        return [
            Cache::STATS_HITS             => $info['keyspace_hits'],
            Cache::STATS_MISSES           => $info['keyspace_misses'],
            Cache::STATS_UPTIME           => $info['uptime_in_seconds'],
            Cache::STATS_MEMORY_USAGE     => $info['used_memory'],
            Cache::STATS_MEMORY_AVAILABLE => false,
        ];
    }

    /**
     * Returns the serializer constant to use. If Redis is compiled with
     * igbinary support, that is used. Otherwise the default PHP serializer is
     * used.
     *
     * @return integer One of the Redis::SERIALIZER_* constants
     */
    protected function getSerializerValue()
    {
        if (defined('Redis::SERIALIZER_IGBINARY') && extension_loaded('igbinary')) {
            return \RedisCluster::SERIALIZER_IGBINARY;
        }

        return \RedisCluster::SERIALIZER_PHP;
    }
}
