<?php

namespace AlloCine\Doctrine\CacheRedisCluster\Cache;

use Csa\Bundle\GuzzleBundle\Cache\DoctrineAdapter;
use Psr\Http\Message\RequestInterface;
use Psr\Http\Message\ResponseInterface;

/**
 * As the allocine API always returns a 200 OK status code even on errors,
 * this class handles the specific caching strategy.
 *
 * A response is considered OK to cache if the status is 200 and there is a non-zero
 * shared max age. If max age is omitted the response is also cached.
 */
class CacheAdapter extends DoctrineAdapter
{
    /**
     * @param RequestInterface  $request
     * @param ResponseInterface $response
     */
    public function save(RequestInterface $request, ResponseInterface $response)
    {
        $statusCode = $response->getStatusCode() ;

        if ($this->getMaxAge($response) === 0) {
            return;
        }

        if ($statusCode >= 200 && $statusCode < 300) {
            parent::save($request, $response);
        }
    }

    /**
     * @param ResponseInterface $response
     *
     * @return integer|null
     */
    private function getMaxAge(ResponseInterface $response)
    {
        if (!$response->hasHeader('Cache-Control')) {
            return null;
        }

        $cacheControl = explode(',', $response->getHeader('Cache-Control')[0]);
        $cacheControl = array_map('trim', $cacheControl);

        $cacheValues = [];
        foreach ($cacheControl as $entry) {
            if (strpos($entry, '=') !== false) {
                $entry = explode('=', $entry);
                $cacheValues[strtolower($entry[0])] = $entry[1];
            } else {
                $cacheValues[strtolower($entry)] = true;
            }
        }

        if (!array_key_exists('max-age', $cacheValues)) {
            return null;
        }

        return intval($cacheValues['max-age']);
    }
}
