<?php

namespace Allocine\DbzModelBundle\Tests\Model\Executive;

use Allocine\DbzModelBundle\Model\Executive\Company;
use Allocine\DbzModelBundle\Model\Executive\CompanyModel;
use Allocine\DbzModelBundle\Model\Tool\Origin;
use Allocine\DbzModelBundle\Tests\KernelTestCase;

class CompanyTest extends KernelTestCase
{
    /**
     * @var CompanyModel
     */
    private $model;

    /**
     * Setup before
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->model = $this
            ->getSession()
            ->getModel(CompanyModel::class);
    }

    /**
     * Test Element Model insertOne automation for a non LEGACY safe element
     */
    public function testInsertOne()
    {
        /**
         * Creating a Company without origin
         */

        $company = $this->model->createEntity([
            'id' => 2000000000,
            'name' => 'Company 1',
        ]);

        $this->model->insertOne($company);

        $company = $this->model->findByPK([
            'id' => 2000000000
        ]);

        $this->assertEquals(Origin::CORE, $company->origin);
        $this->assertEquals('Company 1', $company->name);

        /**
         * Creating a Company with non-legacy origin
         */

        $company = $this->model->createEntity([
            'id' => 2000000001,
            'name' => 'Company 1',
            'origin' => Origin::GRAPH_API
        ]);

        $this->model->insertOne($company);

        $company = $this->model->findByPK([
            'id' => 2000000001
        ]);

        $this->assertEquals(Origin::GRAPH_API, $company->origin);
        $this->assertEquals('Company 1', $company->name);

        /**
         * Creating a Company with LEGACY Origin (which is not allowed)
         */

        $company = $this->model->createEntity([
            'id' => 2000000003,
            'name' => 'Company 3',
            'origin' => Origin::LEGACY
        ]);

        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage(
            'LEGACY elements can\'t be inserted in dbz using pomm.'
        );

        $this->model->insertOne($company);
    }

    /**
     * Test Element Model updateByPk automation for a non LEGACY safe element
     */
    public function testUpdateByPk()
    {
        /**
         * Non legacy element should update updated_at column whatsoever
         */

        $company = $this->model->findByPK([
            'id' => 2000000000
        ]);

        $previousUpdatedAt = $company->updated_at;

        $company->name = 'Company 2000000000';

        $this->model->updateOne($company, ['name']);

        $company = $this->model->findByPK([
            'id' => 2000000000
        ]);

        $this->assertEquals('Company 2000000000', $company->name);
        $this->assertGreaterThan($previousUpdatedAt, $company->updated_at);

        /**
         * Legacy element should not update updated_at column if not safe
         */

        $company = $this->model->findByPK([
            'id' => 2000000001
        ]);

        $previousUpdatedAt = $company->updated_at;

        $company->name = 'Company 2000000001';

        $this->model->updateOne($company, ['name']);

        $company = $this->model->findByPK([
            'id' => 2000000001
        ]);

        $this->assertEquals('Company 2000000001', $company->name);
        $this->assertEquals($previousUpdatedAt, $company->updated_at);

        /**
         * Change origin to LEGACY and updated_at is not allowed.
         */

        $company = $this->model->findByPK([
            'id' => 2000000002
        ]);

        $company->updated_at = new \DateTime();
        $company->origin = Origin::LEGACY;

        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage(
            'updated_at column should never be set for LEGACY elements.'
        );

        $company = $this->model->updateOne($company, [ 'updated_at', 'origin' ]);

        /**
         * Legacy element should not update updated_at column if not safe
         * even if the updated_column is modified in php, it should be ignored
         */


        $company = $this->model->findByPK([
            'id' => 2000000003
        ]);

        $previousUpdatedAt = $company->updated_at;

        $company->name = 'Company 2000000003';
        $company->updated_at = new \DateTime();

        $this->model->updateOne($company, [ 'name', 'updated_at' ]);

        $company = $this->model->findByPK([
            'id' => 2000000003
        ]);

        $this->assertEquals('Company 2000000003', $company->name);
        $this->assertEquals($previousUpdatedAt, $company->updated_at);
    }
}
