<?php

namespace Allocine\DbzModelBundle\Tests\Domain;

use Allocine\DbzModelBundle\Domains\Tag3D;
use Allocine\DbzModelBundle\Domains\Tag3DCollection;
use Allocine\DbzModelBundle\Tests\KernelTestCase;

class Tag3DCollectionTest extends KernelTestCase
{
    public function testConstruct()
    {
        // Test empty collection from PG string

        $tags = new Tag3DCollection('{}');

        $this->assertEquals([], $tags->toStringArray());

        // Complete test

        $tags = new Tag3DCollection([
            [
                'category' => 'A',
                'subcategory' => 'B',
                'tag' => 'C',
            ],
            (object)[
                'category' => 'D',
                'subcategory' => 'E',
                'tag' => 'F',
            ],
            '{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }',
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $this->assertEquals('A', $tags[0]->category);
        $this->assertEquals('B', $tags[0]->subcategory);
        $this->assertEquals('C', $tags[0]->tag);

        $this->assertEquals('D', $tags[1]->category);
        $this->assertEquals('E', $tags[1]->subcategory);
        $this->assertEquals('F', $tags[1]->tag);

        $this->assertEquals('G', $tags[2]->category);
        $this->assertEquals('H', $tags[2]->subcategory);
        $this->assertEquals('I', $tags[2]->tag);

        $this->assertEquals('J', $tags[3]->category);
        $this->assertEquals('K', $tags[3]->subcategory);
        $this->assertEquals('L', $tags[3]->tag);

        $this->assertEquals('M', $tags[4]->category);
        $this->assertEquals('N', $tags[4]->subcategory);
        $this->assertEquals('O', $tags[4]->tag);
    }

    /**
     *
     */
    public function testToStringArray()
    {
        $tags = new Tag3DCollection([
            [
                'category' => 'A',
                'subcategory' => 'B',
                'tag' => 'C',
            ],
            (object)[
                'category' => 'D',
                'subcategory' => 'E',
                'tag' => 'F',
            ],
            '{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }',
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $this->assertEquals([
            'A.B.C',
            'D.E.F',
            'G.H.I',
            'J.K.L',
            'M.N.O'
        ], $tags->toStringArray());
    }

    /**
     *
     */
    public function testContains()
    {
        $tags = new Tag3DCollection([
            [
                'category' => 'A',
                'subcategory' => 'B',
                'tag' => 'C',
            ],
            (object)[
                'category' => 'D',
                'subcategory' => 'E',
                'tag' => 'F',
            ],
            '{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }',
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $this->assertTrue($tags->contains(new Tag3D('A.B.C')));
        $this->assertTrue($tags->contains(new Tag3D('D.E.F')));
        $this->assertTrue($tags->contains(new Tag3D('G.H.I')));
        $this->assertTrue($tags->contains(new Tag3D('J.K.L')));
        $this->assertTrue($tags->contains(new Tag3D('M.N.O')));

        $this->assertFalse($tags->contains(new Tag3D('1.B.C')));
        $this->assertFalse($tags->contains(new Tag3D('2.E.F')));
        $this->assertFalse($tags->contains(new Tag3D('3.H.I')));
        $this->assertFalse($tags->contains(new Tag3D('4.K.L')));
        $this->assertFalse($tags->contains(new Tag3D('5.N.O')));
    }

    /**
     *
     */
    public function testAppend()
    {
        $tags = new Tag3DCollection();

        $tags->append([
            'category' => 'A',
            'subcategory' => 'B',
            'tag' => 'C',
        ]);
        $tags->append((object)[
            'category' => 'D',
            'subcategory' => 'E',
            'tag' => 'F',
        ]);
        $tags->append('{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }'
        );
        $tags->append('J.K.L');
        $tags->append(['M', 'N', 'O']);

        $this->assertTrue($tags->contains(new Tag3D('A.B.C')));
        $this->assertTrue($tags->contains(new Tag3D('D.E.F')));
        $this->assertTrue($tags->contains(new Tag3D('G.H.I')));
        $this->assertTrue($tags->contains(new Tag3D('J.K.L')));
        $this->assertTrue($tags->contains(new Tag3D('M.N.O')));

        $this->assertEquals(new Tag3D('A.B.C'), $tags[0]);
        $this->assertEquals(new Tag3D('D.E.F'), $tags[1]);
        $this->assertEquals(new Tag3D('G.H.I'), $tags[2]);
        $this->assertEquals(new Tag3D('J.K.L'), $tags[3]);
        $this->assertEquals(new Tag3D('M.N.O'), $tags[4]);

        $this->assertCount(5, $tags);

        // Appending existing tags doesn't appends anything

        $tags->append('J.K.L');
        $tags->append(['M', 'N', 'O']);

        $this->assertCount(5, $tags);
    }

    /**
     *
     */
    public function testRemove()
    {
        $tags = new Tag3DCollection([
            [
                'category' => 'A',
                'subcategory' => 'B',
                'tag' => 'C',
            ],
            (object)[
                'category' => 'D',
                'subcategory' => 'E',
                'tag' => 'F',
            ],
            '{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }',
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $tags->remove('A.B.C');

        $this->assertFalse($tags->contains(new Tag3D('A.B.C')));
        $this->assertTrue($tags->contains(new Tag3D('D.E.F')));
        $this->assertTrue($tags->contains(new Tag3D('G.H.I')));
        $this->assertTrue($tags->contains(new Tag3D('J.K.L')));
        $this->assertTrue($tags->contains(new Tag3D('M.N.O')));

        $this->assertEquals(4, count($tags));

        $tags->remove(['M', 'N', 'O']);

        $this->assertFalse($tags->contains(new Tag3D('A.B.C')));
        $this->assertTrue($tags->contains(new Tag3D('D.E.F')));
        $this->assertTrue($tags->contains(new Tag3D('G.H.I')));
        $this->assertTrue($tags->contains(new Tag3D('J.K.L')));
        $this->assertFalse($tags->contains(new Tag3D('M.N.O')));

        $this->assertEquals(3, count($tags));

        // Removing a non existing tag doesn't change the collection

        $tags->remove('1.2.3');

        $this->assertEquals(3, count($tags));
    }

    /**
     *
     */
    public function testMerge()
    {
        $tags = new Tag3DCollection();

        $tags->merge([
            [
                'category' => 'A',
                'subcategory' => 'B',
                'tag' => 'C',
            ],
            (object)[
                'category' => 'D',
                'subcategory' => 'E',
                'tag' => 'F',
            ],
            '{
                "category": "G",
                "subcategory": "H",
                "tag": "I"
            }',
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $this->assertTrue($tags->contains(new Tag3D('A.B.C')));
        $this->assertTrue($tags->contains(new Tag3D('D.E.F')));
        $this->assertTrue($tags->contains(new Tag3D('G.H.I')));
        $this->assertTrue($tags->contains(new Tag3D('J.K.L')));
        $this->assertTrue($tags->contains(new Tag3D('M.N.O')));

        $this->assertEquals(new Tag3D('A.B.C'), $tags[0]);
        $this->assertEquals(new Tag3D('D.E.F'), $tags[1]);
        $this->assertEquals(new Tag3D('G.H.I'), $tags[2]);
        $this->assertEquals(new Tag3D('J.K.L'), $tags[3]);
        $this->assertEquals(new Tag3D('M.N.O'), $tags[4]);

        $this->assertCount(5, $tags);

        // Appending existing tags doesn't appends anything

        $tags->merge([
            'J.K.L',
            ['M', 'N', 'O']
        ]);

        $this->assertCount(5, $tags);
    }

    /**
     *
     */
    public function testMatch()
    {
        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection(['A.B.C', 'A.B.D', 'A.D.E']),
            $tags->match('A.*')
        );
        $this->assertEquals(
            new Tag3DCollection(['A.B.C', 'A.B.D']),
            $tags->match('A.B.*')
        );
        $this->assertEquals(
            new Tag3DCollection(['A.B.C', '1.2.C']),
            $tags->match('*.C')
        );
        $this->assertEquals(
            new Tag3DCollection(['A.B.C', 'A.B.D', 'B.B.B']),
            $tags->match('*.B.*')
        );

    }

    /**
     *
     */
    public function testEmpty()
    {
        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(new Tag3DCollection(), $tags->empty());
    }

    /**
     *
     */
    public function testRemoveFromMatch()
    {
        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection([ 'B.B.B', '1.2.C' ]),
            $tags->removeFromMatch('A.*')
        );

        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection([ 'A.D.E', 'B.B.B', '1.2.C' ]),
            $tags->removeFromMatch('A.B.*')
        );

        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection([ 'A.B.D', 'A.D.E', 'B.B.B' ]),
            $tags->removeFromMatch('*.C')
        );

        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection([ 'A.D.E', '1.2.C' ]),
            $tags->removeFromMatch('*.B.*')
        );

        $tags = new Tag3DCollection([
            'A.B.C',
            'A.B.D',
            'A.D.E',
            'B.B.B',
            '1.2.C'
        ]);

        $this->assertEquals(
            new Tag3DCollection([]),
            $tags->removeFromMatch('*')
        );
    }
}
