<?php

namespace Allocine\DbzModelBundle\Internal\Model;

use Allocine\DbzModelBundle\Model\Tool\Origin;
use PommProject\ModelManager\Model\CollectionIterator;
use PommProject\ModelManager\Model\FlexibleEntity\FlexibleEntityInterface;
use PommProject\ModelManager\Model\Model;
use PommProject\ModelManager\Model\ModelTrait\WriteQueries;
use PommProject\ModelManager\Model\Projection;

abstract class ElementModel extends Model
{
    /**
     * Basics elements still updated from MSSQL can't have their updated_at
     * columns modified.
     *
     * Some elements, however, are no longuer MSSQL dependant and can be
     * updated. In this case, set this constant to true for child classes.
     */
    const LEGACY_SAFE = false;

    /**
     * All elements have writeQueries trait, but updated_at is automatically set
     * at the current timestamp
     */

    use WriteQueries {
        insertOne as _insertOne;
        updateByPk as _updateByPk;
    }

    /**
     * @inheritDoc
     */
    public function insertOne(FlexibleEntityInterface &$entity)
    {
        if (
            isset($entity->extract()['origin']) &&
            Origin::LEGACY === $entity->extract()['origin']
        ) {
            throw new \RuntimeException(
                "LEGACY elements can't be inserted in dbz using pomm."
            );
        }

        if (
            (
                isset($entity->extract()['origin']) &&
                is_null($entity->extract()['origin'])
            ) ||
            ! isset($entity->extract()['origin'])
        ) {
            $entity->hydrate(['origin' => Origin::CORE]);
        }

        return $this->_insertOne($entity);
    }

    /**
     * @inheritDoc
     */
    public function updateByPk(array $primary_key, array $updates)
    {
        /**
         * First thing first : if origin is set to LEGACY, the updated_at field
         * should NOT be set if LEGACY_SAFE is false!
         */

        if (
            ! static::LEGACY_SAFE &&
            isset($updates['origin']) &&
            Origin::LEGACY === $updates['origin'] &&
            isset($updates['updated_at'])
        ) {
            throw new \RuntimeException(
                'updated_at column should never be set for LEGACY elements.'
            );
        } elseif (
            isset($updates['origin']) &&
            Origin::LEGACY !== $updates['origin']
        ) {
            $nonLegacyOrigin = true;
        } else {
            $nonLegacyOrigin = false;
        }

        $updates['updated_at'] = new \DateTime();

        /** Now it's time for the basic update */

        $where = $this
            ->checkPrimaryKey($primary_key)
            ->getWhereFrom($primary_key)
        ;
        $parameters = $this->getParametersList($updates);
        $update_strings = [];

        foreach ($updates as $field_name => $new_value) {
            if (! static::LEGACY_SAFE && 'updated_at' === $field_name) {

                /**
                 * We still want to be sure updated_at is not updated for
                 * LEGACY elements
                 */

                $update_strings[] = sprintf(
                    "%s = CASE
                        WHEN ((NOT %s) AND (origin = 'LEGACY')) THEN updated_at
                        ELSE %s 
                    END",
                    $this->escapeIdentifier($field_name),
                    ($nonLegacyOrigin ? 'TRUE' : 'FALSE'),
                    $parameters[$field_name]
                );
            } else {
                $update_strings[] = sprintf(
                    "%s = %s",
                    $this->escapeIdentifier($field_name),
                    $parameters[$field_name]
                );
            }
        }

        $sql = strtr(
            "UPDATE :relation SET :update WHERE :condition RETURNING :projection",
            [
                ':relation'   => $this->getStructure()->getRelation(),
                ':update'     => join(', ', $update_strings),
                ':condition'  => (string) $where,
                ':projection' => $this->createProjection()->formatFieldsWithFieldAlias(),
            ]
        );

        $iterator = $this->query(
            $sql,
            array_merge(array_values($updates), $where->getValues())
        );

        if ($iterator->isEmpty()) {
            return null;
        }

        return $iterator->current()->status(FlexibleEntityInterface::STATUS_EXIST);
    }

    /**
     * @inheritDoc
     */
    public function query($sql, array $values = [], Projection $projection = null)
    {
        return parent::query($sql, $values, $projection);
    }
}
