<?php

namespace Allocine\DbzModelBundle\Command;

use Allocine\DbzModelBundle\Fixtures\FixtureBackendInterface;
use Allocine\DbzModelBundle\Fixtures\FixtureObject;
use Allocine\DbzModelBundle\Fixtures\FixtureProcessorInterface;
use Nelmio\Alice\Loader\NativeLoader;
use PommProject\ModelManager\Model\Model;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;

class FixturesLoadCommand extends Command
{
    /**
     * @var \ArrayObject
     */
    private $backends;

    /**
     * @var \ArrayObject
     */
    private $processors;

    /**
     * @var string
     */
    private $fixturesPath;

    /**
     * @param string $fixturesPath
     */
    public function __construct(string $fixturesPath)
    {
        parent::__construct();

        $this->fixturesPath = $fixturesPath;
    }

    /**
     * Registers a backend the command will be able to use to persit a fixture.
     *
     * @param string $name
     * @param        $backend
     */
    public function setFixtureBackends(string $name, $backend)
    {
        $this->backends[$name] = $backend;
    }

    /**
     * @param FixtureProcessorInterface $processor
     */
    public function addProcessor(FixtureProcessorInterface $processor)
    {
        $this->processors[get_class($processor)] = $processor;
    }

    /**
     * {@inheritdoc}
     */
    protected function configure(): void
    {
        $this
            ->setName('dbz:fixtures:load')
            ->setDescription('Loads fixtures.')
            ->addArgument('path', InputArgument::OPTIONAL, 'Path to fixtures')
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $loader = new NativeLoader();

        if ($path = $input->getArgument('path')) {
            if (preg_match('/.*\.yaml$/', $path)) {
                $in = dirname($path);
                $name = basename($path);
            } else {
                $in = $path;
                $name = '*.yaml';
            }
        } else {
            $in = $this->fixturesPath;
            $name = '*.yaml';
        }

        /** @var SplFileInfo[] $fixtures */
        $fixtures = (new Finder())
            ->depth('< 3')
            ->in($in)
            ->files()
            ->name($name)
            ->sortByName();

        foreach ($fixtures as $fixture) {
            $output->writeln(sprintf('Loading "<info>%s/%s</info>"', basename($fixture->getPath()), $fixture->getFilename()));

            /** @var FixtureObject $object */
            foreach ($loader->loadFile($fixture->getRealPath())->getObjects() as $object) {
                $this->doPersist($object, $output);
            }
        }

        return 0;
    }

    /**
     * @param FixtureObject   $object
     * @param OutputInterface $output
     *
     * @throws \PommProject\ModelManager\Exception\ModelException
     */
    private function doPersist(FixtureObject $object, OutputInterface $output)
    {
        try {
            $backend = $this->getBackend($object);
            $object->doPersist($backend, $this->processors);
        } catch (\Exception $e) {
            $output->writeln(sprintf(<<<EOF
<error>

  Exception thrown while trying to persist fixture.

  Exception:            %s
  Fixture:              %s
  Backend service id:   %s
  Backend class:        %s

  Session (Pomm models only): %s
</error>
EOF
                , get_class($e), get_class($object), $object::getBackendServiceId(), is_object($backend) ? get_class($backend) : '', $backend instanceof Model ? $backend->getSession()->getStamp() : ''));
            throw $e;
        }
    }

    /**
     * @param FixtureObject $object
     *
     * @return Model|FixtureBackendInterface
     *
     * @throws \Exception
     */
    private function getBackend(FixtureObject $object)
    {
        try {
            return $this->backends[$object::getBackendServiceId()];
        } catch (\Exception $e) {
            throw new \Exception(sprintf(<<<EOF
Unable to use backend [%1\$s] to persist object of type [%2\$s].

Please make sure service [%1\$s] is registered to the command by calling %3\$s->setFixtureBackends(\$id, new Reference(\$id))
during the dependency injection process.
EOF
                , $object::getBackendServiceId(), get_class($object), self::class));
        }
    }
}
