# Fixtures

The bundle provides a simple way to insert fixtures, using the `nelmio/alice` package.

- [Loading fixtures](#loading)
- [Defining fixtures](#fixtures-definition)
- [Persisting - persist() override](#override-persist)
- [Persisting - using processors](#fixture-processors)

## <a name="loading"></a>Loading fixtures

The command:

```bash
$ bin/console dbz:fixture:load --help
Description:
  Loads fixtures.

Usage:
  dbz:fixtures:load [<path>]

Arguments:
  path                  Path to fixtures
```

By default, the system will look for fixtures into `projectDir/fixtures` and can be changed with:

```yaml
dbz_model:
  fixtures_path: '%kernel.project_dir%/somewhere_else
```

Or by passing the `path` argument to the command (useful to process a single fixture file).

```bash
$ bin/console dbz:fixture:load fixtures/subpath/a_fixture.yaml
```

## <a name="fixtures-definition"></a>Fixtures definition

A fixture file must start with the name of the class which will be used to define and persist the fixture:

```yaml
App\Token\TokenFixture:
  anonymous_www:
    id: 1000
    id_application: 34
```

The fixture class must extends `FixtureObject`:

```php
<?php

namespace App\Token;

use Allocine\DbzModelBundle\Fixtures\FixtureObject;
use Allocine\DbzModelBundle\Model\Secure\TokenModel;

class TokenFixture extends FixtureObject
{
    /**
     * {@inheritdoc}
     */
    public static function getBackendServiceId(): string
    {
        return TokenModel::class;
    }

    public static function registeredProcessors(): array
    {
    }
}
```

You must define which service/backend will be used to persist the fixture by implementing `getBackendServiceId()`.

Every pomm models are already registered to the command. If for some reason you need another backend service, the
service will need to implement `FixtureBackendInterface` and must be tagged with `dbz.fixture_backend`.

## Overriding the way fixture are imported

### <a name="override-persist"></a>The FixtureObject::persist() method

The first solution is to override the `persist()` method:

```php
<?php

namespace App\Token;

use Allocine\DbzModelBundle\Fixtures\FixtureObject;
use Allocine\DbzModelBundle\Model\Secure\TokenModel;

class TokenFixture extends FixtureObject
{
    /**
     * {@inheritdoc}
     */
    public static function getBackendServiceId(): string
    {
        return TokenModel::class;
    }

    public static function registeredProcessors(): array
    {
    }

    protected function persist($model)
    {
        $this['something'] = 'value';

        parent::persist($model);
    }
}

```

This is simple, but not re-usable, unless you refactor this into an abstract class your fixture definition will extend.

### <a name="fixture-processors"></a>Using fixture processors

Fixture processors are services that will be used to pre-preprocess the fixture before it is inserted, so:

- a single processor can be used to alter different fixture types,
- a fixture can be pre-processed by multiple processors (making processors re-usable).

They must implement `Allocine\DbzModelBundle\Fixtures\FixtureProcessorInterface`:

```php
<?php

namespace Allocine\DbzModelBundle\Fixtures;

interface FixtureProcessorInterface
{
    /**
     * @param FixtureObject $object
     *
     * @return mixed
     */
    public function process(FixtureObject $object);
}
```

Fixture processors are added to the command service by tagging them with `dbz.fixture_processor`

Re-implementing our previous example would then look like this:

```yaml
#services.yaml
services:
    App\Token\TokenFixtureProcessor:
        tags:
            - { name: dbz.fixture_processor }
```

```php
<?php

namespace App\Token;

use Allocine\DbzModelBundle\Fixtures\FixtureObject;
use Allocine\DbzModelBundle\Fixtures\FixtureProcessorInterface;

class TokenFixtureProcessor implements FixtureProcessorInterface
{
    public function __construct(...$youParams)
    {
        // your business logic and other injected services, required to process the object
    }

    public function process(FixtureObject $object)
    {
        $object['something'] = 'value';
    }
}
```