<?php

namespace Allocine\DbzModelBundle\Internal\Converter;

use Allocine\DbzModelBundle\Domains\Tag3D as Tag3DDomain;
use Allocine\DbzModelBundle\Domains\Tag3DCollection;
use PommProject\Foundation\Converter\ArrayTypeConverter;
use PommProject\Foundation\Exception\ConverterException;
use PommProject\Foundation\Exception\FoundationException;
use PommProject\Foundation\Session\Session;

/**
 * Tag3DArray converter
 *
 * Convert a PostgreSQL tag3d[] array to a Tag3DCollection object and vice versa.
 *
 * Originally this class was extending PommProject\Foundation\Converter\PgArray but due to the need of a custom fromPg return type, to return directly a Tag3DCollection,
 * it was modified to extend directly ArrayTypeConverter and re-implement the needed methods.
 * Not the most elegant solution but it works...
 *
 * @see Tag3DCollection
 * @see Tag3D
 */
class Tag3DArray extends ArrayTypeConverter
{
    /**
     * @inheritdoc
     */
    public static function getSubType(string $type): string
    {
        return 'public.tag3d';
    }

    /**
     * @inheritdoc
     */
    public function fromPg($data, string $type, Session $session): Tag3DCollection
    {
        $tag3DArray = $this->fromPgBase($data, $type, $session);

        return (new Tag3DCollection($tag3DArray));
    }

    /**
     * PgArray base implementation of fromPg
     *
     * @throws FoundationException
     * @see ConverterInterface
     */
    public function fromPgBase(?string $data, string $type, Session $session): ?array
    {
        if ($data === null || $data === 'NULL') {
            return null;
        }

        $type = static::getSubType($type);

        if ($data !== "{}") {
            $converter = $this->getSubtypeConverter($type, $session);

            return array_map(function ($val) use ($converter, $type, $session) {
                if ($val !== "NULL") {
                    return null !== $val && preg_match('/\\\\/', $val)
                        ? $converter->fromPg(stripslashes($val), $type, $session)
                        : $converter->fromPg($val, $type, $session);
                }

                return null;
            }, str_getcsv(trim($data, "{}")));
        }

        return [];
    }

    /**
     * @inheritdoc
     */
    public function toPgStandardFormat($data, string $type, Session $session): ?string
    {
        $tags = [];

        foreach ($data as $tag) {
            $tags[] = (new Tag3DDomain($tag))->toLtreeArray();
        }

        return $this->toPgStandardFormatBase($tags, $type, $session);
    }

    /**
     * PgArray base implementation of toPgStandardFormat
     *
     * @throws FoundationException
     * @see ConverterInterface
     */
    private function toPgStandardFormatBase(mixed $data, string $type, Session $session): ?string
    {
        if ($data === null) {
            return null;
        }

        $type = static::getSubType($type);
        $converter = $this->getSubtypeConverter($type, $session);
        $data = $this->checkArray($data);

        return
            sprintf('{%s}', implode(',',
                array_map(function ($val) use ($converter, $type, $session) {
                    if ($val === null) {
                        return 'NULL';
                    }

                    $val = $converter->toPgStandardFormat($val, $type, $session);

                    if ($val !== '') {
                        if (preg_match('/[,\\"\s]/', $val)) {
                            $val = sprintf('"%s"', addcslashes($val, '"\\'));
                        }
                    } else {
                        $val = '""';
                    }

                    return $val;
                }, $data)
            ));
    }

    /**
     * @throws FoundationException
     * @see ConverterInterface
     */
    public function toPg(mixed $data, string $type, Session $session): string
    {
        $type = static::getSubType($type);

        if ($data === null) {
            return sprintf("NULL::%s[]", $type);
        }

        $converter = $this->getSubtypeConverter($type, $session);
        $data = $this->checkArray($data);

        return sprintf(
            'ARRAY[%s]::%s[]',
            implode(',', array_map(fn($val) => $converter->toPg($val, $type, $session), $data)),
            $type
        );
    }

    protected function checkArray(mixed $data): array
    {
        if (!is_array($data)) {
            try {
                $data = $data->toArray();
            } catch (\Exception $e) {
                throw new ConverterException(
                    sprintf("Tag3DArray converter data must be an array or a Tag3DCollection occurrence ('%s' given).", gettype($data))
                );
            }
        }

        return $data;
    }
}
