<?php

namespace Allocine\DbzModelBundle\Tests\Domain;

use Allocine\DbzModelBundle\Domains\Tag3D;
use Allocine\DbzModelBundle\Tests\KernelTestCase;

class Tag3DTest extends KernelTestCase
{
    public function testConstruct()
    {
        // Basic titania construct from array

        $tag = new Tag3D([
            'category' => 'A',
            'subcategory' => 'B',
            'tag' => 'C',
        ]);

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Basic titania construct from object

        $tag = new Tag3D((object) [
            'category' => 'A',
            'subcategory' => 'B',
            'tag' => 'C',
        ]);

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Basic titania construct from json string

        $tag = new Tag3D('{
            "category": "A",
            "subcategory": "B",
            "tag": "C"
        }');

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Tag from pg string representation

        $tag = new Tag3D('A.B.C');

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Tag from pomm array representation

        $tag = new Tag3D(['A', 'B', 'C']);

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Tag from old Tag3D constructor

        $tag = new Tag3D('A', 'B', 'C');

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Tag from json basic array, just in case

        $tag = new Tag3D('[ "A", "B", "C"]');

        $this->assertEquals('A', $tag->category);
        $this->assertEquals('B', $tag->subcategory);
        $this->assertEquals('C', $tag->tag);

        // Invalid data

        $this->expectException('\Exception');
        $this->expectExceptionMessage(
            'Invalid $data for Allocine\DbzModelBundle\Domains\Tag3D constructor'
        );

        new Tag3D(2);
    }

    public function testToString()
    {
        $tag = new Tag3D('A', 'B', 'C');

        $this->assertEquals('A.B.C', $tag->__toString());
    }

    public function testToLtreeString()
    {
        $tag = new Tag3D('A', 'B', 'C');

        $this->assertEquals(['A', 'B', 'C'], $tag->toLtreeArray());
    }

    public function testSetAttribute()
    {
        $tag = new Tag3D('Youpla boum', 'coin   pan', 'ohéohé');

        $this->assertEquals('YouplaBoum', $tag->category);
        $this->assertEquals('CoinPan', $tag->subcategory);
        $this->assertEquals('Oheohe', $tag->tag);

        $tag = new Tag3D('youpla-@@/Boum', '#\\"coin __  pan', 'ohéàùè');

        $this->assertEquals('YouplaBoum', $tag->category);
        $this->assertEquals('CoinPan', $tag->subcategory);
        $this->assertEquals('Oheaue', $tag->tag);
    }

    public function testMatch()
    {
        $tag = new Tag3D('A', 'B', 'C');

        $this->assertTrue($tag->match('A.B.C'));
        $this->assertTrue($tag->match('A.B.*'));
        $this->assertTrue($tag->match('A.*'));
        $this->assertTrue($tag->match('*'));
        $this->assertTrue($tag->match('*.B.*'));
        $this->assertTrue($tag->match('*.B.C'));
        $this->assertTrue($tag->match('*.C'));

        $this->assertFalse($tag->match('A'));
        $this->assertFalse($tag->match('*.1'));
        $this->assertFalse($tag->match('2.*'));
        $this->assertFalse($tag->match('B.*'));
        $this->assertFalse($tag->match('*.A'));
    }

    public function testIsEqualTo()
    {
        $tag = new Tag3D('A', 'B', 'C');

        $this->assertTrue($tag->isEqualTo('A.B.C'));
        $this->assertTrue($tag->isEqualTo(['A', 'B', 'C']));
        $this->assertTrue($tag->isEqualTo(new Tag3D('A.B.C')));
        $this->assertTrue($tag->isEqualTo(new Tag3D('A', 'B', 'C')));

        $this->assertFalse($tag->isEqualTo('D.E.F'));
        $this->assertFalse($tag->isEqualTo(new Tag3D('D.E.F')));
    }
}
