<?php

namespace Allocine\DbzModelBundle\Tests\Model\Social;

use Allocine\DbzModelBundle\Model\Social\User;
use Allocine\DbzModelBundle\Model\Social\UserModel;
use Allocine\DbzModelBundle\Model\Tool\Origin;
use Allocine\DbzModelBundle\Tests\KernelTestCase;

class UserTest extends KernelTestCase
{
    /**
     * @var UserModel
     */
    private $model;

    /**
     * Setup before
     */
    protected function setUp(): void
    {
        parent::setUp();

        $this->model = $this
            ->getSession()
            ->getModel(UserModel::class);
    }

    /**
     * Test Element Model insertOne automation for a LEGACY safe element
     */
    public function testInsertOne()
    {
        /**
         * Creating a User without origin
         */

        $user = $this->model->createEntity([
            'id_brand' => 'AC',
            'id' => 2000000000
        ]);

        $this->model->insertOne($user);

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000000
        ]);

        $this->assertEquals(Origin::CORE, $user->origin);

        /**
         * Creating a User with non-legacy origin
         */

        $user = $this->model->createEntity([
            'id_brand' => 'AC',
            'id' => 2000000001,
            'origin' => Origin::GRAPH_API
        ]);

        $this->model->insertOne($user);

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000001
        ]);

        $this->assertEquals(Origin::GRAPH_API, $user->origin);

        /**
         * Creating a user with LEGACY Origin (which is not allowed)
         */

        $user = $this->model->createEntity([
            'id_brand' => 'AC',
            'id' => 2000000003,
            'origin' => Origin::LEGACY
        ]);

        $this->expectException(\RuntimeException::class);
        $this->expectExceptionMessage(
            'LEGACY elements can\'t be inserted in dbz using pomm.'
        );

        $this->model->insertOne($user);
    }

    /**
     * Test Element Model updateByPk automation for a LEGACY safe element
     */
    public function testUpdateByPk()
    {
        /**
         * Non Legacy element should update updated_at column whatsoever
         */

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000000
        ]);

        $previousUpdatedAt = $user->updated_at;

        $user->email = 'coin1@pan.com';

        $this->model->updateOne($user, [ 'email' ]);

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000000
        ]);

        $this->assertEquals('coin1@pan.com', $user->email);
        $this->assertGreaterThan($previousUpdatedAt, $user->updated_at);

        /**
         * Legacy element should update updated_at column because it's a safe
         * class
         */

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000001
        ]);

        $previousUpdatedAt = $user->updated_at;

        $user->email = 'coin2@pan.com';

        $this->model->updateOne($user, [ 'email' ]);

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000001
        ]);

        $this->assertEquals('coin2@pan.com', $user->email);
        $this->assertGreaterThan($previousUpdatedAt, $user->updated_at);

        /**
         * Because User is LEGACY_SAFE, we can even change the origin to LEGACY
         * if we wish to.
         */

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000002
        ]);

        $previousUpdatedAt = $user->updated_at;

        $user->origin = Origin::LEGACY;
        $user->email = 'coin3@pan.com';

        $this->model->updateOne($user, [ 'email', 'origin' ]);

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000002
        ]);

        $this->assertEquals('coin3@pan.com', $user->email);
        $this->assertEquals(Origin::LEGACY, $user->origin);
        $this->assertGreaterThan($previousUpdatedAt, $user->updated_at);
    }

    /**
     * Test if the mainAvatar is the good one
     */
    public function testGetMainAvatar()
    {
        foreach (range(2000000000, 2000000003) as $id) {

            /** @var User $user */

            $user = $this->model->findByPK([
                'id_brand' => 'AC',
                'id' => $id
            ]);

            $this->assertEquals(
                "http://domain/$id.jpg",
                $user->getMainAvatar()
            );
        }

        $user = $this->model->findByPK([
            'id_brand' => 'AC',
            'id' => 2000000004
        ]);

        $this->assertNull($user->getMainAvatar());
    }
}
