<?php

namespace Allocine\DbzModelBundle\Model\Social;

use Allocine\DbzModelBundle\Common\Helper\OrderHelper;
use Allocine\DbzModelBundle\Internal\Model\BrandAwareModelInterface;
use Allocine\DbzModelBundle\Internal\Model\BrandAwareModelTrait;
use Allocine\DbzModelBundle\Internal\Model\ElementModel;
use Allocine\DbzModelBundle\Internal\Model\ModelTrait;
use Allocine\DbzModelBundle\Model\Social\AutoStructure\User as UserStructure;
use PommProject\Foundation\Exception\SqlException;
use PommProject\Foundation\Where;
use PommProject\ModelManager\Model\Model;

/**
 * UserModel.
 *
 * Model class for table user.
 *
 * @see Model
 */
class UserModel extends ElementModel implements BrandAwareModelInterface
{
    use ModelTrait;
    use BrandAwareModelTrait;

    const LEGACY_SAFE = true;

    /** @var array */
    private static $userDeps = [];

    /**
     * __construct().
     *
     * Model constructor
     */
    public function __construct()
    {
        $this->structure = new UserStructure();
        $this->flexible_entity_class = User::class;
    }

    /**
     * @param string $email
     * @param string $idBrand
     *
     * @return null|User
     */
    public function loadByEmail(string $email, string $idBrand): ?User
    {
        $iterator = $this->findWhere('email = $* AND id_brand = $* AND active', [$email, $idBrand]);

        if (!$iterator->isEmpty()) {
            return $iterator->current();
        }

        return null;
    }

    /**
     * @return array|mixed|null
     */
    public function loadByActiveGid(string $gid)
    {
        return $this->loadBy('gid = $* AND active', [$gid]);
    }

    /**
     * @return array|mixed|null
     */
    public function loadByGid(string $gid)
    {
        return $this->loadBy('gid = $*', [$gid]);
    }

    /**
     * @return array|mixed|null
     */
    private function loadBy(string $filters, array $parameters)
    {
        $iterator = $this->findWhere($filters, $parameters);

        if (!$iterator->isEmpty()) {
            return $iterator->current();
        }

        return null;
    }

    /**
     * @param string $idLegacy
     * @return User|null
     */
    public function loadByIdLegacy(string $idLegacy): ?User
    {
        return $this->findWhere(
            'id_legacy = $*',
            [ 'id_legacy = ' => $idLegacy ]
        )->current();
    }

    /**
     * @param string $field
     * @param array $ids
     * @param array $fields
     * @param string $suffix
     *
     * @return array
     */
    public function findAllWhereIn(
        string $field,
        array $ids,
        array $fields = [],
        string $suffix = ''
    ): array {
        $where = Where::createWhereIn($field, $ids);

        $connection = $this->getSession()->getConnection();

        $sql = strtr("
            SELECT {projection}
            FROM {relation}
            WHERE {where}
            ORDER BY {order} $suffix",
            [
                '{projection}' => $this->createProjection($fields),
                '{relation}' => $this->structure->getRelation(),
                '{where}' => $where,
                '{order}' => OrderHelper::buildFixedOrder($connection, $field, $ids),
            ]
        );

        return iterator_to_array($this->query($sql, $where->getValues()));
    }

    /**
     * @param User $user
     * @return mixed
     */
    protected function anonymize(User $user)
    {
        $data = [
            'nickname' => null,
            'profile' => [],
            'email' => null,
            'active' => false,
            'password' => null,
            'salt' => null,
            'alt' => [],
        ];

        $user->hydrate($data);

        return $this->updateOne($user, array_keys($data));
    }

    /**
     * @param User $user
     */
    public function deleteOrAnonymize(User $user)
    {
        if ($this->canDelete($user)) {
            $this->deleteOne($user);
        } else {
            $this->anonymize($user);
        }
    }

    /**
     * @param User $user
     * @return bool
     * @throws SqlException
     */
    public function canDelete(User $user): bool
    {
        $connection = $this->getSession()->getConnection();
        $tables = $this->getSocialUserDependencies();

        $query = "SELECT 1 from %s where %s";
        foreach ($tables as $table => $constraints) {
            foreach ($constraints as $columns) {
                $where = [];
                foreach ($columns as $column) {
                    // knowing that the constraint is composed of 2 columns (id_user and id_brand)
                    if ($column == 'id_brand') {
                        $where[] = "id_brand = '$user->id_brand'";
                    } else {
                        $where[] = "$column = $user->id";
                    }
                }

                if ($connection->executeAnonymousQuery(sprintf($query, $table, implode(' AND ', $where)))->countRows() > 0) {
                    return false;
                }
            }
        }

        return true;
    }

    private function getSocialUserDependencies(): array
    {
        if (!empty(self::$userDeps)) {
            return self::$userDeps;
        }

        $connection = $this->getSession()->getConnection();
        $tables = [];

        $query = <<<'SQL'
SELECT DISTINCT tc.table_schema, tc.constraint_name, tc.table_name, kcu.column_name
FROM information_schema.table_constraints tc
JOIN information_schema.key_column_usage kcu ON tc.constraint_name = kcu.constraint_name
JOIN information_schema.constraint_column_usage ccu ON ccu.constraint_name = tc.constraint_name
WHERE constraint_type = 'FOREIGN KEY'
AND ccu.table_schema='social'
AND ccu.table_name='user'
AND tc.table_name<>'user';
SQL;

        $results = $connection->executeAnonymousQuery($query);

        for($i=0; $i < $results->countRows(); $i++) {
            $item = $results->fetchRow($i);
            $tables[$item['table_schema'].'.'.$item['table_name']][$item['constraint_name']][] = $item['column_name'];
        }

        self::$userDeps = $tables;

        return self::$userDeps;
    }
}
