<?php

namespace Allocine\DbzModelBundle\Fixtures;

/**
 * A base object that allows any field to be set with a setter
 * of the form `set*`. (for instance setName, setId ...).
 *
 * @codeCoverageIgnore
 */
abstract class FixtureObject extends \ArrayObject
{
    /**
     * The id of the service which will be used to import a fixture.
     *
     * @return string
     */
    abstract public static function getBackendServiceId(): string;

    /**
     * A list of FixtureProcessorInterface instances which will be applied to a fixture.
     *
     * @return array
     */
    abstract public static function registeredProcessors(): array;

    /**
     * @codeCoverageIgnore
     *
     * @param $model
     */
    protected function persist($model)
    {
        $model->createAndSave($this->getArrayCopy());
    }

    /**
     * @param string $name
     * @param array  $value
     *
     * @throws \RuntimeException If the called function is not a setter
     *
     * @return mixed
     */
    public function __set($name, $value)
    {
        return $this[$name] = $value;
    }

    /**
     * @param                                   $model
     * @param array|FixtureProcessorInterface[] $processors
     *
     * @throws \Exception
     */
    final public function doPersist($model, $processors = [])
    {
        foreach ($this->registeredProcessors() as $processorName) {
            if (!array_key_exists($processorName, $processors)) {
                throw new \Exception(
                    sprintf(
                        'Processor \'%s\' is required by \'%s\', but is not registered.',
                        $processorName,
                        get_class($this)
                    )
                );
            }
            $processors[$processorName]->process($this);
        }

        $this->persist($model);
    }
}
