<?php

namespace Allocine\DbzModelBundle\Intervention;

use Allocine\DbzModelBundle\Model\Media\Image;
use Allocine\DbzModelBundle\Model\Media\ImageModel;
use Allocine\DbzModelBundle\Model\Tool\Origin;
use Intervention\Image\Exception\NotReadableException;
use Intervention\Image\ImageManager as InterventionImageManager;
use League\Flysystem\Config;
use League\Flysystem\Local\LocalFilesystemAdapter;
use League\Flysystem\Filesystem;
use League\Flysystem\Visibility;
use PommProject\Foundation\Exception\SqlException;

class ImageManager
{
    const LOCAL_TMP_FILE = '/tmp/image';

    const BASE_URL = 'http://fr.web.img1.acsta.net';

    const MIME_TO_FILE_EXTENSION = [
        'image/gif' => 'gif',
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
    ];

    /**
     * @var InterventionImageManager
     */
    private $manager;

    /**
     * @var \Intervention\Image\Image
     */
    private $image;

    /**
     * @var ImageModel
     */
    private $imageModel;

    /**
     * @var string
     */
    private $imageBasePath;

    /**
     * @var string
     */
    private $cacheDir;

    /**
     * @var Filesystem
     */
    private $imgFileSystem;

    /**
     * ImageManager constructor.
     *
     * @param ImageModel $imageModel
     * @param string $imagesRoot
     * @param string $imageBasePath
     * @param string $cacheDir
     */
    public function __construct(
        ImageModel $imageModel,
        string $imagesRoot,
        string $imageBasePath,
        string $cacheDir
    ) {
        $this->imageModel = $imageModel;
        $this->imageBasePath = $imageBasePath;
        $this->cacheDir = $cacheDir;
        $this->manager = new InterventionImageManager(['driver' => 'imagick']);
        $this->imgFileSystem = new Filesystem(
            new LocalFilesystemAdapter($imagesRoot)
        );
    }

    /**
     * @param string $source
     * @param array $type
     * @param bool $autoSave
     * @param string $origin
     * @param int|null $idUser
     * @param bool|$b2bAvailable
     *
     * @return mixed
     */
    public function uploadImage(
        string $source,
        array $type,
        bool $autoSave = true,
        string $origin = Origin::WWM_IMPORT,
        int $idUser = null,
        bool $b2bAvailable = false
    )
    {
        /*
         * At first we get the image and copy it locally
         */
        $this->image = $this->manager->make($source);

        if (!is_dir($this->cacheDir . DIRECTORY_SEPARATOR . 'image')) {
            mkdir($this->cacheDir . DIRECTORY_SEPARATOR . 'image');
        }

        $tmpFilePath = $this->cacheDir .
            DIRECTORY_SEPARATOR . 'image' .
            DIRECTORY_SEPARATOR . uniqid('image_', true) .
            '.' .
            pathinfo($source, PATHINFO_EXTENSION);

        if ($this->image->extension === 'gif' || $this->image->mime === 'image/gif') {
            copy($source, $tmpFilePath);
        } else {
            $this->image->save($tmpFilePath);
        }


        $checksum = md5_file($tmpFilePath);

        /*
         * Check if image already exists
         */

        if ($image = $this->checksumExists($checksum)) {
            // no need to create image, delete temp file and return
            unlink($tmpFilePath);

            return [
                'image' => $image,
                'exists' => true,
                'created' => false,
            ];
        }

        /**
         * If not, we create the data for the Image entity
         */
        $data = [
            'active' => true,
            'origin' => $origin,
            'created_at' => new \DateTime(),
            'updated_at' => new \DateTime(),
            'id_created_by' => $idUser,
            'b2b_available' => $b2bAvailable,
            'tags' => [$type],
            'checksum' => $checksum,
            'path' => sprintf(
                '/%s/%s/%s/%s.%s',
                $this->imageBasePath,
                substr($checksum, 0, 2), // first 2 chars of checksum
                substr($checksum, 2, 2), // 3-4 chars of checksum
                $checksum,
                isset(self::MIME_TO_FILE_EXTENSION[$this->image->mime()]) ?
                    self::MIME_TO_FILE_EXTENSION[$this->image->mime()] : 'jpg'
            ),
            'data' => [
                'file' => [
                    'size' => $this->image->filesize(),
                    'width' => $this->image->width(),
                    'height' => $this->image->height(),
                    'mime' => $this->image->mime(),
                ],
            ],
        ];

        /*
         * We copy it to the media server
         */

        $stream = fopen($tmpFilePath, 'r+');

        $this->imgFileSystem->writeStream(
            str_replace($this->imageBasePath, '', $data['path']),
            $stream,
            [Config::OPTION_DIRECTORY_VISIBILITY => Visibility::PUBLIC]
        );

        if (is_resource($stream)) {
            fclose($stream);
        }

        unlink($tmpFilePath);

        /**
         * And we save the image in the database
         */
        $image = $autoSave ? $this->imageModel->createAndSave($data) : $this->imageModel->createEntity($data);

        return [
            'image' => $image,
            'exists' => $autoSave,
            'created' => $autoSave,
        ];
    }

    /**
     * @param string $checksum
     *
     * @return mixed
     */
    private function checksumExists(string $checksum)
    {
        return $this
            ->imageModel
            ->findWhere('checksum = $*', [$checksum])
            ->current()
        ;
    }

    /**
     * @param Image $image
     *
     * @return Image
     */
    public function updateImageData(Image $image): Image
    {
        try {
            $localImage = $this->manager->make(self::BASE_URL . $image->path);
        } catch (NotReadableException $e) {
            $image->active = false;

            return $image;
        }

        $localImage->save(self::LOCAL_TMP_FILE);

        $image->checksum = md5_file(self::LOCAL_TMP_FILE);

        $data = $image->data;

        $data['file'] = [
            'size' => $localImage->filesize() ?: filesize(self::LOCAL_TMP_FILE),
            'width' => $localImage->width(),
            'height' => $localImage->height(),
            'mime' => $localImage->mime(),
        ];

        $image->data = $data;

        return $image;
    }

    public function replaceAndDelete(Image $fromImage, Image $toImage)
    {
        // Get tables referencing media.image(id)
        //
        // Yeah, it's a fun query inside pg internals. May be we should work on
        // a generalization for this one.

        $sql = <<<SQL
            SELECT
              conrelid::pg_catalog.regclass distant_table,
              a.attname distant_column
            FROM pg_catalog.pg_class c
            LEFT JOIN pg_catalog.pg_namespace n
              ON n.oid = c.relnamespace
            INNER JOIN pg_catalog.pg_constraint cn
              ON cn.confrelid = c.oid
              AND cn.contype = 'f'
            INNER JOIN pg_attribute a
              ON a.attrelid = conrelid
              AND a.attnum = cn.conkey[1]
            INNER JOIN pg_attribute a2
              ON a2.attrelid = confrelid
              AND ARRAy[a2.attnum] = cn.confkey
            WHERE c.relname = 'image'
            AND n.nspname = 'media'
            AND a2.attname = 'id'
SQL;
        $linkedTables = $this->imageModel->getSession()->getQueryManager()->query($sql);

        foreach ($linkedTables as $linkedTable) {
            try {
                $this->imageModel->getSession()->getQueryManager()->query(
                    strtr(
                        'UPDATE :relname SET :column = $* WHERE :column = $*',
                        [
                            ':relname' => $linkedTable['distant_table'],
                            ':column' => $linkedTable['distant_column'],
                        ]
                    ),
                    [
                        $toImage->id,
                        $fromImage->id,
                    ]
                );
            } catch (SqlException $e) {
                if ($e->getSQLErrorState() !== SqlException::UNIQUE_VIOLATION) {
                    throw $e;
                }

                // When replacing duplicated images, it is possible the new data
                // already exists... In this case, no use to update data, we just remove it

                $this->imageModel->getSession()->getQueryManager()->query(
                    strtr(
                        'DELETE FROM :relname WHERE :column = $*',
                        [
                            ':relname' => $linkedTable['distant_table'],
                            ':column' => $linkedTable['distant_column'],
                        ]
                    ),
                    [
                        $fromImage->id,
                    ]
                );
            }
        }

        $this->imageModel->deleteOne($fromImage);
    }
}
