<?php

namespace Allocine\DbzModelBundle\Internal\Model;

use PommProject\Foundation\Where;
use PommProject\ModelManager\Exception\ModelException;
use PommProject\ModelManager\Model\CollectionIterator;
use PommProject\ModelManager\Model\FlexibleEntity\FlexibleEntityInterface;
use PommProject\ModelManager\Model\Model;
use PommProject\ModelManager\Model\ModelTrait\ReadQueries;

/**
 * PartitionWriteQueries
 *
 * Rewrite write queries for model instances.
 */
trait PartitionWriteQueries
{
    /**
     * insertOne
     *
     * Insert a new entity in the database. The entity is passed by reference.
     * It is updated with values returned by the database (ie, default values).
     *
     * @access public
     * @param  FlexibleEntityInterface  $entity
     * @return Model                    $this
     */
    public function insertOne(FlexibleEntityInterface &$entity)
    {
        $values = $entity->fields(
            array_intersect(
                array_keys($this->getStructure()->getDefinition()),
                array_keys($entity->fields())
            )
        );
        $sql = strtr(
            "insert into :relation (:fields) values (:values)",
            [
                ':relation'   => $this->getStructure()->getRelation(),
                ':fields'     => $this->getEscapedFieldList(array_keys($values)),
                ':values'     => join(',', $this->getParametersList($values))
            ]);

        $this->query($sql, array_values($values));

        $primaryKeys = $this->getStructure()->getPrimaryKey();
        $primaryValues = [];

        foreach ($primaryKeys as $primaryKey) {
            $primaryValues[$primaryKey] = $values[$primaryKey];
        }

        $entity = $this->findByPK($primaryValues);

        return $this;
    }

    /**
     * updateByPk
     *
     * Update a record and fetch it with its new values. If no records match
     * the given key, null is returned.
     *
     * @access public
     * @param  array          $primary_key
     * @param  array          $updates
     * @throws ModelException
     * @return FlexibleEntityInterface
     */
    public function updateByPk(array $primary_key, array $updates)
    {
        $where = $this
            ->checkPrimaryKey($primary_key)
            ->getWhereFrom($primary_key)
        ;
        $parameters = $this->getParametersList($updates);
        $update_strings = [];

        foreach ($updates as $field_name => $new_value) {
            $update_strings[] = sprintf(
                "%s = %s",
                $this->escapeIdentifier($field_name),
                $parameters[$field_name]
            );
        }

        $sql = strtr(
            "update :relation set :update where :condition",
            [
                ':relation'   => $this->getStructure()->getRelation(),
                ':update'     => join(', ', $update_strings),
                ':condition'  => (string) $where,
            ]
        );

        $this->query($sql, array_merge(array_values($updates), $where->getValues()));

        return $this->findByPK($primary_key);
    }

    /**
     * deleteWhere
     *
     * Delete records by a given condition. A collection of all deleted entries is returned.
     *
     * @param        $where
     * @param  array $values
     * @return CollectionIterator
     */
    public function deleteWhere($where, array $values = [])
    {
        if (!$where instanceof Where) {
            $where = new Where($where, $values);
        }

        $entities = $this->findWhere($where);

        $sql = strtr(
            "delete from :relation where :condition",
            [
                ':relation'   => $this->getStructure()->getRelation(),
                ':condition'  => (string) $where,
            ]
        );

        $collection = $this->query($sql, $where->getValues());
        foreach ($entities as $entity) {
            $entity->status(FlexibleEntityInterface::STATUS_NONE);
        }
        $entities->rewind();

        return $entities;
    }
}
