<?php

namespace Allocine\DbzModelBundle\Domains;

use Allocine\DbzModelBundle\Common\Helper\StringHelper;
use Allocine\DbzModelBundle\Common\Helper\TransliteratorHelper;
use Allocine\Titania\Type\Base\ConstrainedObject;

/**
 * @property string category
 * @property string subcategory
 * @property string tag
 */
class Tag3D extends ConstrainedObject
{
    const IGNORED_CHAR_PATTERN = '/[^A-Za-z0-9\s]/';

    /**
     * @return array
     */
    public static function getAttributeDefinition()
    {
        return [
            'category' => null,
            'subcategory' => null,
            'tag' => null,
        ];
    }

    /**
     * Tag3D constructor.
     * @throws \Exception
     */
    public function __construct()
    {
        $args = func_get_args();

        if (count($args) === 1) {

            if (is_string($args[0])) {
                // We have a string. It could be a json object (handled by
                // titania), a json array or a Tag3D string witch must be
                // handled locally

                $jsonData = json_decode($args[0]);

                if (is_null($jsonData)) {
                    $args = $this->getArgsFromTag3DFromString($args[0]);
                } elseif (is_array($jsonData)) {
                    $args = $this->getArgsFromTag3DFromSimpleArray($jsonData);
                } else {
                    $args = $args[0];
                }
            } elseif (is_array($args[0])) {

                // If we have a 3-elements array with only keys being 0, 1 and 2
                // then we should have a Tag3D from pomm ltree handling

                if (
                    count($args[0]) === 3 &&
                    array_key_exists(0, $args[0]) &&
                    array_key_exists(1, $args[0]) &&
                    array_key_exists(2, $args[0])
                ) {
                    $args = $this->getArgsFromTag3DFromSimpleArray($args[0]);
                } else {
                    $args = $args[0];
                }
            } else {
                $args = $args[0];
            }
        } elseif (count($args) === 3) {
            // in this case, we obviously have 3 arguments like old Tag3d
            // construtor. In this case, if all values are strings, we construct
            // an array of values

            if (
                is_string($args[0]) &&
                is_string($args[1]) &&
                is_string($args[2])
            ) {
                $args = $this->getArgsFromTag3DFromSimpleArray($args);
            } else {
                throw new \Exception(
                    'Invalid $args for ' . $this->getClass() . ' constructor'
                );
            }
        }

        parent::__construct($args);
    }

    /**
     * @param string $string
     * @return array
     */
    private function getArgsFromTag3DFromString(string $string): array
    {
        return $this->getArgsFromTag3DFromSimpleArray(
            explode('.', $string)
        );
    }

    /**
     * @param string $string
     * @return array
     */
    private function getArgsFromTag3DFromSimpleArray(array $array): array
    {
        return [
            'category' => $array[0],
            'subcategory' => $array[1],
            'tag' => $array[2],
        ];
    }

    public function isEqualTo($other): bool
    {
        if (is_string($other) || is_array($other)) {
            $other = new self($other);
        }

        if (is_object($other) && $other instanceof self) {
            return
                $this->category === $other->category &&
                $this->subcategory === $other->subcategory &&
                $this->tag === $other->tag;
        }

        return false;
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return implode('.', [
            $this->category,
            $this->subcategory,
            $this->tag
        ]);
    }

    /**
     * @return string[]
     */
    public function toLtreeArray(): array
    {
        return [$this->category, $this->subcategory, $this->tag];
    }

    /**
     * set an property while validating and transforming data
     *
     * @param string $name  Attribute name
     * @param mixed  $value Attribute value
     *
     * @throws \Exception
     *
     * @return mixed
     */

    public function setAttribute($name, $value)
    {
        $value = str_replace(' ', '', ucwords(preg_replace(
            static::IGNORED_CHAR_PATTERN,
            '',
            StringHelper::stripAccents($value)
        )));

        return parent::setAttribute($name, $value);
    }

    /**
     * @param string $mask
     * @return bool
     */
    public function match(string $mask): bool
    {
        return fnmatch($mask, $this->__toString());
    }
}
