<?php

namespace Allocine\DbzModelBundle\Domains;

use Allocine\Titania\Type\Base\ConstrainedObject;

/**
 * @property string schema
 * @property string table
 * @property string brand
 * @property string id
 */

class GlobalId extends ConstrainedObject
{
    const NODE_EPISODE = 'Episode';
    const NODE_SEASON = 'Season';
    const NODE_SERIES = 'Series';
    const NODE_MOVIE = 'Movie';
    const NODE_PERSON = 'Person';
    const NODE_NEWS = 'News';
    const NODE_FESTIVAL = 'Festival';
    const NODE_FESTIVAL_EDITION = 'FestivalEdition';
    const NODE_USER = 'User';
    const NODE_USER_REVIEW = 'UserReview';
    const NODE_VIDEO = 'Video';

    /**
     * @return array
     */
    public static function getAttributeDefinition()
    {
        return [
            'schema' => null,
            'table' => null,
            'brand' => null,
            'id' => null
        ];
    }

    /**
     * GlobalId constructor.
     * @throws \Exception
     */
    public function __construct()
    {
        $args = func_get_args();

        if (count($args) === 1) {

            if (is_string($args[0])) {
                // We have a string. It could be a json object (handled by
                // titania), a json array or a GlobalId string witch must be
                // handled locally

                $jsonData = json_decode($args[0]);

                if (is_null($jsonData)) {
                    $args = $this->getArgsFromGlobalIdFromString($args[0]);
                } elseif (is_array($jsonData)) {
                    $args = $this->getArgsFromGlobalIdFromSimpleArray($jsonData);
                } else {
                    $args = $args[0];
                }
            } elseif (is_array($args[0])) {

                // If we have a 3-elements array with only keys being 0, 1 and 2
                // then we should have a GlobalId from pomm ltree handling

                if (count($args[0]) === 4 &&
                    array_key_exists(0, $args[0]) &&
                    array_key_exists(1, $args[0]) &&
                    array_key_exists(2, $args[0]) &&
                    array_key_exists(3, $args[0])
                ) {
                    $args = $this->getArgsFromGlobalIdFromSimpleArray($args[0]);
                } else {
                    $args = $args[0];
                }
            } else {
                $args = $args[0];
            }
        } elseif (count($args) === 4) {
            // in this case, we obviously have 4 arguments like old GlobalId
            // construtor. In this case, if all values are strings, we construct
            // an array of values

            if (is_string($args[0]) &&
                is_string($args[1]) &&
                is_string($args[2]) &&
                is_string($args[3])
            ) {
                $args = $this->getArgsFromGlobalIdFromSimpleArray($args);
            } else {
                throw new \Exception(
                    'Invalid $args for ' . $this->getClass() . ' constructor'
                );
            }
        }

        parent::__construct($args);
    }

    /**
     * @param string $string
     * @return array
     */
    private function getArgsFromGlobalIdFromString(string $string) : array
    {
        return $this->getArgsFromGlobalIdFromSimpleArray(
            explode('.', $string)
        );
    }

    /**
     * @param string $string
     * @return array
     */
    private function getArgsFromGlobalIdFromSimpleArray(array $array) : array
    {
        return [
            'schema' => $array[0],
            'table' => $array[1],
            'brand' => $array[2],
            'id' => $array[3],
        ];
    }

    /**
     * @return string
     */
    public function __toString(): string
    {
        return sprintf(
            '%s.%s.%s.%s',
            $this->schema,
            $this->table,
            $this->brand,
            $this->id
        );
    }

    /**
     * @return string[]
     */
    public function toLtreeArray(): array
    {
        return [$this->schema, $this->table, $this->brand, $this->id];
    }

    /**
     * set an property while validating and transforming data
     *
     * @param string $name  Attribute name
     * @param mixed  $value Attribute value
     *
     * @throws \Exception
     *
     * @return mixed
     */

    public function setAttribute($name, $value)
    {
        // Special case for brand. Can only be a two-uppercase letter string
        // or an underscore character

        if (($name === 'brand') && ! is_null($value)) {
            if (! preg_match('/^[A-Z]{2}$/', $value) && $value !== '_' ) {
                throw new \Exception(
                    'Invalid brand value [' . $value . '] for ' .
                    $this->getClass()
                );
            }
        } else {
            $value = str_replace(' ', '', preg_replace(
                "/[^A-Za-z0-9_\s]/",
                '',
                $value
            ));
        }

        parent::setAttribute($name, $value);
    }

    /**
     * @param string $mask
     * @return bool
     */
    public function match(string $mask): bool
    {
        return fnmatch($mask, $this->__toString());
    }

    /**
     * @return string
     */
    public function getNodeType(): string
    {
        switch ($this->table) {
            case 'series_episode':
                return self::NODE_EPISODE;
            case 'series_season':
                return self::NODE_SEASON;
            case 'series':
            case 'program':
                return self::NODE_SERIES;
            case 'movie':
                return self::NODE_MOVIE;
            case 'person':
                return self::NODE_PERSON;
            case 'news':
                return self::NODE_NEWS;
            case 'festival':
                return self::NODE_FESTIVAL;
            case 'festival_edition':
                return self::NODE_FESTIVAL_EDITION;
            case 'user':
                return self::NODE_USER;
            case 'user_has_review':
                return self::NODE_USER_REVIEW;
            case 'video_legacy':
            case 'video':
                return self::NODE_VIDEO;
            default:
                throw new \RuntimeException(sprintf(
                    'Could not read Node type : Unsupported gid "%s".',
                    $this->__toString()
                ));
        }
    }
}
