# Symfony services in dbz-model-bundle

## Models

All the models from the bundle are available as **private** services.

To get the list, just use `bin/console debug:container --tag=pomm.model --show-private`.

Every model services is aliased like this `<PREFIX>.<RELATION_NAME>`:

* `PREFIX` is `dbz.model` by default
* `RELATION_NAME` is the table name, ex: `secure.token`
* so we get something like `dbz.model.secure.token` as an alias for the `TokenModel` service 

The prefix eases integration in other project:

```
$ bin/console debug:container --show-private | grep dbz.model

  dbz.model.brand.entity_has_override           alias for "Allocine\DbzModelBundle\Model\Brand\EntityHasOverrideModel"
  dbz.model.country.entity_has_custom_data      alias for "Allocine\DbzModelBundle\Model\Country\EntityHasCustomDataModel"
  dbz.model.country.entity_has_override         alias for "Allocine\DbzModelBundle\Model\Country\EntityHasOverrideModel"
  dbz.model.executive.company                   alias for "Allocine\DbzModelBundle\Model\Executive\CompanyModel"
  dbz.model.executive.company_localized         alias for "Allocine\DbzModelBundle\Model\Executive\CompanyLocalizedModel"
  dbz.model.executive.country                   alias for "Allocine\DbzModelBundle\Model\Executive\CountryModel"
  dbz.model.executive.locale                    alias for "Allocine\DbzModelBundle\Model\Executive\LocaleModel"
  dbz.model.executive.localization              alias for "Allocine\DbzModelBundle\Model\Executive\LocalizationModel"
  dbz.model.media.image                         alias for "Allocine\DbzModelBundle\Model\Media\ImageModel"
  dbz.model.media.image_localized               alias for "Allocine\DbzModelBundle\Model\Media\ImageLocalizedModel"
  dbz.model.movie.movie                         alias for "Allocine\DbzModelBundle\Model\Movie\MovieModel"
  dbz.model.movie.movie_has_company             alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasCompanyModel"
  dbz.model.movie.movie_has_country             alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasCountryModel"
  dbz.model.movie.movie_has_global_box_office   alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasGlobalBoxOfficeModel"
  dbz.model.movie.movie_has_image               alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasImageModel"
  dbz.model.movie.movie_has_person              alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasPersonModel"
  dbz.model.movie.movie_has_person_localized    alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasPersonLocalizedModel"
  dbz.model.movie.movie_has_release             alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasReleaseModel"
  dbz.model.movie.movie_has_soundtrack          alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasSoundtrackModel"
  dbz.model.movie.movie_has_temporal_box_office alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasTemporalBoxOfficeModel"
  dbz.model.movie.movie_has_video               alias for "Allocine\DbzModelBundle\Model\Movie\MovieHasVideoModel"
  dbz.model.movie.movie_localized               alias for "Allocine\DbzModelBundle\Model\Movie\MovieLocalizedModel"
  dbz.model.movie.movie_version                 alias for "Allocine\DbzModelBundle\Model\Movie\MovieVersionModel"
  dbz.model.secure.application                  alias for "Allocine\DbzModelBundle\Model\Secure\ApplicationModel"
  dbz.model.secure.token                        alias for "Allocine\DbzModelBundle\Model\Secure\TokenModel"
  dbz.model.secure.user                         alias for "Allocine\DbzModelBundle\Model\Secure\UserModel"
```

By default, the prefix will be: `dbz.model`.

## Read/write concerns

Having models able to fallback on a specific `Session` depending on the query type (`SELECT` vs `UPDATE/INSERT`) is a bit 
complex at this time: for exemple, a complex transaction having `SELECT` and `INSERT` at the same time will mean having a 
model able to detect this and falling back to a specific POMM `Session`.

The developer will then have to choose on a backend depending on what he is going to do.

When a service `App\A` requires the model corresponding to the `movie.movie` table, and when it's for read-only:

```
services:
    App\A:
        arguments: ["@dbz.model.movie.movie.slave"]
```

When it comes to write, then use the alias or the service fully qualified name:

```
services:
    App\A:
        arguments:
            - Allocine\DbzModelBundle\Model\Movie\MovieModel
# or
    App\A:
        arguments: ["@dbz.model.movie.movie"]
```

**Important regarding autowiring**

Since the main model is registered to the master Session, using autowiring without specifying the arguments will make
your service A using the master backend by default.

So again, for a service A:

```php
namespace App;

class A
{
    private $model;

    public function __construct(Allocine\DbzModelBundle\Model\Movie\MovieModel $model)
    {
        $this->model = $model;
    }
}
```

Read-only dependent service will be registered like this:

```
# config.yaml
services:
    App\A:
        arguments: ["@dbz.model.movie.movie.slave"]

```

While read/write dependent services can use autowiring like this:

```
# config.yaml
services:
    App\A:
        autowire: true
```

To force a slave service while others stay with the master service:

```yml
  AppBundle\Command\Import\Wwm\MovieImportCommand:
    autowire: true
    arguments:
      $countryModel: "@dbz.model.executive.country.slave"
    tags:
      - { name: 'console.command' }
```

Doing this will force the country model to use the slave session.
