<?php

namespace Allocine\DbzModelBundle\DependencyInjection;

use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Extension\PrependExtensionInterface;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

class DbzModelExtension extends Extension implements PrependExtensionInterface
{
    /**
     * @param array            $configs
     * @param ContainerBuilder $container
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $loader = new YamlFileLoader($container, new FileLocator(__DIR__.'/../Resources/config'));
        $loader->load('services.yaml');

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        $this->configure($config, $container);
    }

    /**
     * @param array            $config
     * @param ContainerBuilder $container
     */
    public function configure(array $config, ContainerBuilder $container)
    {
        $container->setParameter('dbz_model.service_prefix', $config['service_prefix']);
    }

    /**
     * Prepends the configuration for the pomm bundle.
     * Configure one master and one slave session as it should be under:
     * pomm:
     *     configuration:
     *         a:
     *             dsn:
     *             ....
     *         b:
     *             dsn:
     *             ....
     *
     * @param ContainerBuilder $container
     */
    public function prepend(ContainerBuilder $container)
    {
        $pommConfig = [
            'configuration' => [],
        ];

        $dbzModelConfig = $container->getExtensionConfig('dbz_model')[0];

        if (count($dbzModelConfig['backends']) < 2) {
            $this->throwConfigurationError(
                sprintf('You\'ve provided %s backends to the DbzModelBundle, 2 are required.', count($dbzModelConfig['backends'])),
                true
            );
        }

        $sessionBuilderClass = 'Allocine\DbzModelBundle\Internal\Core\SessionBuilder';
        $slaveSessionBuilderClass = 'Allocine\DbzModelBundle\Internal\Core\SlaveSessionBuilder';

        $masterSession = $masterSessionName = null;
        foreach ($dbzModelConfig['backends'] as $name => $backendConfig) {
            $sessionName = $name;
            if ($backendConfig['type'] == ServiceHelper::BACKEND_MASTER) {
                if (!isset($backendConfig['name'])) {
                    $this->throwConfigurationError(sprintf(
                        'Master backend \'%s\' requires that you set a name for it.',
                        $name
                    ));
                }

                $masterSession = $name;
                $masterSessionName = $sessionName = $backendConfig['name'];
            } else {
                if (isset($backendConfig['name'])) {
                    throw new InvalidConfigurationException(
                        sprintf('Setting a name for a backend is reserved to master nodes (%s).', $name)
                    );
                }
            }

            $pommSession = [
                'dsn' => $backendConfig['dsn'],
                'pomm:default' => $backendConfig['type'] == ServiceHelper::BACKEND_MASTER,
                'session_builder' => $backendConfig['type'] == ServiceHelper::BACKEND_MASTER ? $sessionBuilderClass : $slaveSessionBuilderClass,
            ];

            $pommConfig['configuration'][$sessionName] = $pommSession;
        }

        if (!$masterSession) {
            $this->throwConfigurationError('You need at least one master session for pomm.', true);
        }

        $slaveCounts = 0;
        foreach ($pommConfig['configuration'] as $name => $configuration) {
            if ($name !== $masterSessionName) {
                ++$slaveCounts;
                unset($pommConfig['configuration'][$name]);
                $slaveSessionName = sprintf('%s.%s_%s', $masterSessionName, ServiceHelper::BACKEND_SLAVE, $slaveCounts);
                $pommConfig['configuration'][$slaveSessionName] = $configuration;
            }
        }

        $container->prependExtensionConfig('pomm', $pommConfig);
    }

    /**
     * Throws a configuration error and display help if needed.
     *
     * @param string $error
     * @param bool   $help
     */
    private function throwConfigurationError(string $error, bool $help = false)
    {
        $helpDoc = $help ? <<<EOF
# Exemple: app/config/config.yaml
dbz_model:
  service_prefix: dbz.model
  backends:
    master:
      dsn: "%database_dsn%"
      type: master
      name: dbz
    slave_1:
      dsn: "%database_dsn%"
      type: slave
EOF
: '';

        throw new InvalidConfigurationException("{$error}\n\n{$helpDoc}");
    }
}
