<?php

namespace Allocine\DbzModelBundle\DependencyInjection;

use Allocine\DbzModelBundle\Command\FixturesLoadCommand;
use Symfony\Component\Config\Definition\Exception\InvalidConfigurationException;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Extension\PrependExtensionInterface;
use Symfony\Component\DependencyInjection\Loader\YamlFileLoader;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

class DbzModelExtension extends Extension implements PrependExtensionInterface
{
    /**
     * @param array            $configs
     * @param ContainerBuilder $container
     *
     * @throws \Exception
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $loader = new YamlFileLoader($container, new FileLocator(__DIR__.'/../Resources/config'));
        $loader->load('services.yaml');

        $configuration = new Configuration($container->getParameter('kernel.project_dir'));
        $config = $this->processConfiguration($configuration, $configs);

        $this->configure($config, $container);

        $container->getDefinition(FixturesLoadCommand::class)
            ->setArgument(0, $config['fixtures_path']);
    }

    /**
     * @param array            $config
     * @param ContainerBuilder $container
     */
    public function configure(array $config, ContainerBuilder $container)
    {
        $encryptor = $container->getDefinition('Allocine\DbzModelBundle\Crypto\SensitiveDataEncryptor');
        $encryptor->setArguments([
            '$key' => $config['sensitive_data_encryption_key'],
            '$algo' => $config['sensitive_data_encryption_algorithm'],
        ]);

        $container->setParameter('dbz_model.force_master_backend', $config['force_master_backend']);
    }

    /**
     * Prepends the configuration for the pomm bundle.
     * Configure one master and one slave session as it should be under:
     * pomm:
     *     configuration:
     *         a:
     *             dsn:
     *             ....
     *         b:
     *             dsn:
     *             ....
     *
     * @param ContainerBuilder $container
     */
    public function prepend(ContainerBuilder $container)
    {
        $pommConfig = [
            'configuration' => [],
        ];

        $dbzModelConfig = $container->getExtensionConfig('dbz_model')[0];

        if (count($dbzModelConfig['backends']) < 2) {
            $this->throwConfigurationError(
                sprintf('You\'ve provided %s backends to the DbzModelBundle, 2 are required.', count($dbzModelConfig['backends'])),
                true
            );
        }

        $sessionBuilderClass = 'Allocine\DbzModelBundle\Internal\Core\SessionBuilder';
        $slaveSessionBuilderClass = 'Allocine\DbzModelBundle\Internal\Core\SlaveSessionBuilder';

        $masterSession = $masterSessionName = null;
        foreach ($dbzModelConfig['backends'] as $name => $backendConfig) {
            if ($backendConfig['type'] == ServiceHelper::BACKEND_MASTER) {
                $sessionName = $masterSession = $name;
            } elseif ($backendConfig['type'] == ServiceHelper::BACKEND_SLAVE) {
                $sessionName = $name;
            } else {
                throw new \Exception(sprintf('Unable to determine backend type for %s '.__METHOD__, $name));
            }

            $pommSession = [
                'dsn' => $backendConfig['dsn'],
                'pomm:default' => $backendConfig['type'] == ServiceHelper::BACKEND_MASTER,
                'session_builder' => $backendConfig['type'] == ServiceHelper::BACKEND_MASTER ? $sessionBuilderClass : $slaveSessionBuilderClass,
            ];

            $pommConfig['configuration'][$sessionName] = $pommSession;
        }

        if (!$masterSession) {
            $this->throwConfigurationError('You need at least one master session for pomm.', true);
        }

        if (array_key_exists('query_logger', $dbzModelConfig)) {
            $pommConfig['logger']['service'] = $dbzModelConfig['query_logger'];
        }

        $container->prependExtensionConfig('pomm', $pommConfig);
    }

    /**
     * Throws a configuration error and display help if needed.
     *
     * @param string $error
     * @param bool   $help
     */
    private function throwConfigurationError(string $error, bool $help = false)
    {
        $helpDoc = $help ? <<<EOF
# Exemple: app/config/config.yaml
dbz_model:
  sensitive_data_encryption_key: "%sensitive_data_encryption_key%"
  sensitive_data_encryption_algorithm: "%sensitive_data_encryption_algorithm%"
  backends:
    master:
      dsn: "%database_dsn%"
      type: master
    slave:
      dsn: "%database_dsn%"
      type: slave
EOF
: '';

        throw new InvalidConfigurationException("{$error}\n\n{$helpDoc}");
    }
}
