<?php

namespace Allocine\DbzModelBundle\Tests\Model;

use Allocine\DbzModelBundle\Tests\App\TestKernel;
use Allocine\DbzModelBundle\Tests\KernelTestCase;
use PommProject\Foundation\Session\ResultHandler;
use PommProject\ModelManager\Model\Model;
use PommProject\ModelManager\Session;
use Symfony\Component\DependencyInjection\ContainerBuilder;

class ConsistencyCheckTest extends KernelTestCase
{
    public function testModelConsistency()
    {
        $kernel = static::bootKernel();

        $builder = \Closure::bind(function () { return $this->buildContainer(); }, $kernel, get_class($kernel));

        /** @var ContainerBuilder $container */
        $container = $builder();
        $container->compile();

        $models = array_map(function ($className) use ($container) {
            return $container->get($className);
        }, array_keys($container->findTaggedServiceIds('pomm.model')));

        foreach ($models as $model) {
            $this->checkModelAgainstDatabase($model, $kernel);
            $this->checkFlexibleEntityExistence($model);
        }
    }

    private function checkModelAgainstDatabase(Model $model, TestKernel $kernel)
    {
        /** @var Session $session */
        $session = $kernel->getContainer()->get('test.pomm_session');
        $query = $session->getPreparedQuery('
            SELECT *
            FROM information_schema.columns WHERE table_schema = $* AND table_name = $*
        ');

        /** @var ResultHandler $results */
        $results = $query->execute(explode('.', $model->getStructure()->getRelation()));

        for ($i = 0; $i < $results->countRows(); ++$i) {
            $fieldDefinition = $results->fetchRow($i);
            $columnName = $fieldDefinition['column_name'];
            $this->checkSameType($model->getStructure()->getRelation(), $model->getStructure()->getTypeFor($columnName), $fieldDefinition);
        }
    }

    private function checkSameType($modelName, $type, $fieldDefinition)
    {
        $columnName = $fieldDefinition['column_name'];
        $dataType = $fieldDefinition['udt_name'];

        switch ($type) {
            case 'public.ltree':
                $this->assertSame('ltree', $fieldDefinition['udt_name']);
                break;
            case 'public.partial_date':
                $this->assertSame('text', $fieldDefinition['udt_name']);
                $this->assertSame('partial_date', $fieldDefinition['domain_name']);
                break;
            case 'public.geometry':
                $this->assertSame('geometry', $fieldDefinition['udt_name']);
                break;
            case 'public.global_id':
                $this->assertSame('ltree', $fieldDefinition['udt_name']);
                $this->assertSame('global_id', $fieldDefinition['domain_name']);
                break;
            case 'tool.origin':
                break;
            case 'public.tag3d_array':
                $this->assertSame('_ltree', $fieldDefinition['udt_name']);
                break;
            case 'public.tag3d':
                $this->assertSame('ltree', $fieldDefinition['udt_name']);
                break;
            case 'public.link':
                $this->assertSame('jsonb', $fieldDefinition['udt_name']);
                break;
            default:
                $this->assertTrue(
                    $type == $dataType,
                    sprintf(
                        'Model \'%s\' expects type \'%s\' for column \'%s\', got \'%s\' in database.',
                        $modelName,
                        $type,
                        $columnName,
                        $dataType
                    )
                );
        }
    }

    /**
     * Prevents having models with a non-existing FlexibleEntity class to be published.
     *
     * @param Model $model
     */
    private function checkFlexibleEntityExistence(Model $model)
    {
        $this->assertTrue(
            class_exists($model->getFlexibleEntityClass()),
            sprintf(
                'Class [%s] configured in [%s] cannot be found.',
                $model->getFlexibleEntityClass(),
                get_class($model)
            )
        );
    }
}
