<?php

namespace Allocine\DbzModelBundle\Crypto;

class SensitiveDataEncryptor
{
    const FIELD_TYPE_STR = 'string';
    const FIELD_TYPE_JSON_STR = 'json';

    /**
     * @var string
     */
    private $key;

    /**
     * @var string
     */
    private $algo;

    /**
     * @param string $key
     * @param string $algo
     */
    public function __construct(string $key, string $algo = 'aes-128-cbc')
    {
        $this->key = $key;
        $this->algo = $algo;
    }

    /**
     * Note that if the given value is a json string, then the second parameter should be equal to the FIELD_TYPE_STR
     * constant and a call back must be provided which will be responsible to escape the string.
     *
     * @param string        $value
     * @param string        $type
     * @param callable|null $callback
     *
     * @return string an hex string
     */
    public function encrypt(string $value, $type = self::FIELD_TYPE_STR, callable $callback = null): string
    {
        if ($type == self::FIELD_TYPE_JSON_STR) {
            if (!$callback) {
                throw new \RuntimeException('You must provide an escape function when encrypting a json string');
            }

            $value = call_user_func($callback, $value);
        }

        return bin2hex(@openssl_encrypt($value, $this->algo, $this->key, OPENSSL_RAW_DATA));
    }

    /**
     * @param mixed  $value value can be a resource, a raw binary string or anything else
     * @param string $type
     *
     * @return string
     */
    public function decrypt($value, $type = self::FIELD_TYPE_STR): string
    {
        $value = is_resource($value) ? stream_get_contents($value) : $value;

        if (!$this->isBinary($value)) {
            $value = hex2bin($value);
        }

        $decrypted = openssl_decrypt($value, $this->algo, $this->key, OPENSSL_RAW_DATA);

        if ($type == self::FIELD_TYPE_JSON_STR) {
            $decrypted = trim($decrypted, "'");
        }

        return $decrypted;
    }

    /**
     * Returns the key used to encrypt data.
     *
     * @return string
     */
    public function getKey(): string
    {
        return $this->key;
    }

    /**
     * Transform a given value into a bytea string (meant to be inserted in the database).
     * If the given value is a binary string, then it's converted to hex.
     *
     * @param string $value
     *
     * @return string
     */
    public function toBytea($value)
    {
        return $this->isBinary($value) ? '\x'.bin2hex($value) : '\x'.$value;
    }

    /**
     * Return true if the given value is a binary string.
     *
     * @param $value
     *
     * @return bool
     */
    public function isBinary($value)
    {
        return preg_match('~[^\x20-\x7E\t\r\n]~', $value) > 0;
    }
}
