<?php

namespace AlloCine\BrandBundle\Kernel;

use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\HttpKernel\Kernel;
use Symfony\Component\Config\Loader\LoaderInterface;

abstract class BrandableKernel extends Kernel
{
    /**
     * @var string
     */
    private $brand;

    /**
     * @param string $environment
     * @param bool $debug
     * @param null $brand
     */
    public function __construct($environment, $debug, $brand = 'allocine')
    {
        $this->brand = $brand;

        parent::__construct($environment, $debug);
    }

    /**
     * @return string
     */
    public function getCacheDir()
    {
        return $this->getProjectDir().'/var/cache/'.$this->getEnvironment().'/'.$this->brand;
    }

    /**
     * @return string
     */
    public function getLogDir()
    {
        return $this->getProjectDir().'/var/log';
    }

    /**
     * @param ContainerBuilder $container
     * @param LoaderInterface  $loader
     *
     * @throws \Exception
     */
    protected function configureContainer(ContainerBuilder $container, LoaderInterface $loader)
    {
        $envconfig = $this->getProjectDir().'/config/config_'.$this->getEnvironment().'.yml';

        if (file_exists($envconfig)) {
            $loader->load($envconfig);
        }

        $brandEnvConfig = $this->getProjectDir().'/config/'.$this->brand.'/config_'.$this->getEnvironment().'.yml';
        $brandConfig = $this->getProjectDir().'/config/'.$this->brand.'/config.yml';

        if (file_exists($brandEnvConfig)) {
            $loader->load($brandEnvConfig);
        } elseif (file_exists($brandConfig)) {
            $loader->load($this->getProjectDir().'/config/'.$this->brand.'/config.yml');
        }
    }

    /**
     * @return string
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param string $brand
     *
     * @throws \LogicException
     */
    public function setBrand($brand)
    {
        if (true === $this->booted) {
            throw new \LogicException('Brand can only be changed when kernel is not booted');
        }

        $this->brand = $brand;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return ucfirst($this->brand) . ($this->isDebug() ? 'Debug' : '' );
    }

    /**
     * @return string
     */
    protected function getContainerClass()
    {
        return $this->name.ucfirst($this->environment).'ProjectContainer';
    }

    /**
     * {@inheritDoc}
     */
    public function serialize()
    {
        return serialize(array($this->environment, $this->debug, $this->brand));
    }

    /**
     * {@inheritDoc}
     */
    public function unserialize($data)
    {
        list($environment, $debug, $brand) = unserialize($data);

        $this->__construct($environment, $debug, $brand);
    }

    /**
     * {@inheritDoc}
     */
    protected function getKernelParameters()
    {
        return array_merge(parent::getKernelParameters(), [
            'brand' => $this->brand
        ]);
    }
}
