<?php

namespace AlloCine\BrandBundle\Model;

class FolderRepository
{
    /**
     * @var array
     */
    private $brands;

    /**
     * @var array
     */
    private $folders;

    /**
     * @var string
     */
    private $defaultBrand;

    /**
     * FolderRepository constructor.
     *
     * @param array       $brands
     * @param array       $folders
     * @param string|null $defaultBrand
     */
    public function __construct(array $brands, array $folders, $defaultBrand = null)
    {
        $this->brands = $brands;
        $this->folders = $folders;
        $this->defaultBrand = $defaultBrand;
    }

    /**
     * @param string $tag
     * @param string $brand
     *
     * @return string[]
     */
    public function getTaggedFolders($tag, $brand = null)
    {
        return array_merge($this->findBrandedFolders($tag, $brand), $this->findCommonFolders($tag));
    }

    /**
     * @param string $tag
     *
     * @return string[]
     */
    public function getAllTaggedFolders($tag)
    {
        $pathes = [];

        foreach ($this->brands as $brand) {
            $pathes = array_merge($pathes, $this->findBrandedFolders($tag, $brand));
        }

        return array_merge($pathes, $this->findCommonFolders($tag));
    }

    /**
     * @param string $tag
     * @param string $brand
     *
     * @throws  \InvalidArgumentException If no provided brand or no default brand set.
     *
     * @return string[]
     */
    protected function findBrandedFolders($tag, $brand = null)
    {
        $pathes = [];

        $brand = $brand ?? $this->defaultBrand;

        if (!$brand) {
            throw new \InvalidArgumentException('No provided brand and no default brand set.');
        }

        foreach ($this->folders as $folder) {
            if (in_array($tag, $folder['tags'])) {
                $pathes[] = str_replace('{brand}', $brand, $folder['path']);
            }
        }

        return $pathes;
    }

    /**
     * @param string $tag
     *
     * @return string[]
     */
    protected function findCommonFolders($tag)
    {
        $pathes = [];

        foreach ($this->folders as $folder) {
            if (in_array($tag, $folder['tags']) && $folder['commonFolder']) {
                $pathes[] = str_replace('{brand}', $folder['commonFolder'], $folder['path']);
            }
        }

        return $pathes;
    }
}
