<?php

namespace AlloCine\BrandBundle\Routing\Loader;

use AlloCine\BrandBundle\Model\BrandRepository;
use AlloCine\BrandBundle\Routing\Extension\LoaderExtensionInterface;
use Symfony\Component\Config\FileLocatorInterface;
use Symfony\Component\Routing\Loader\YamlFileLoader as BaseYamlFileLoader;
use Symfony\Component\Routing\RouteCollection;

/**
 * YAML file loader which can handle branded routes
 *
 * It allows extensions to customize routes options and behavior
 */
class YamlFileLoader extends BaseYamlFileLoader
{
    /**
     * @var BrandRepository
     */
    private $brandRepository;

    /**
     * @var LoaderExtensionInterface[]
     */
    private $extensions;

    /**
     * @param FileLocatorInterface $locator
     * @param BrandRepository      $brandRepository
     */
    public function __construct(FileLocatorInterface $locator, BrandRepository $brandRepository)
    {
        parent::__construct($locator);

        $this->brandRepository = $brandRepository;
        $this->extensions      = [];
    }

    /**
     * @param LoaderExtensionInterface $extension
     */
    public function addExtension(LoaderExtensionInterface $extension)
    {
        $this->extensions[] = $extension;
    }

    /**
     * {@inheritsDoc}
     */
    public function load($file, $type = null)
    {
        $collection = parent::load($file, $type);

        foreach ($this->extensions as $extension) {
            $extension->postLoad($collection);
        }

        return $collection;
    }

    /**
     * @inheritdoc
     *
     * @throws \InvalidArgumentException
     */
    protected function parseRoute(RouteCollection $collection, $name, array $config, $path)
    {
        if (!is_array($config['path'])) {
            parent::parseRoute($collection, $name, $config, $path);

            return;
        }

        foreach ($config['path'] as $brand => $pattern) {
            $brandData = $this->brandRepository->get($brand);
            $brandedRouteName = $brand.'__'.$name;
            $brandConfig = $config;
            $brandConfig['defaults'] = array_merge(
                ['_brand' => $brand],
                isset($config['defaults']) ? $config['defaults'] : []
            );
            $brandConfig['path'] = $pattern;
            if(!isset($config['host'])) {
                $host = $brandData->getHost();
                if(isset($config['options']['branded_host'])) {
                    $host = $brandData->getExtra($config['options']['branded_host']);
                }
                $brandConfig['host'] = $host;
            }
            parent::parseRoute($collection, $brandedRouteName, $brandConfig, $path);
        }
    }

    /**
     * @inheritdoc
     */
    protected function validate($config, $name, $path)
    {
        foreach ($this->extensions as $extension) {
            $extension->validate($config['options'][$extension->getName()] ?? null);
        }

        parent::validate($config, $name, $path);
    }
}
