<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Model;

use InvalidArgumentException;

class FolderRepository
{
    public function __construct(
        private readonly array $brands,
        private readonly array $folders,
        private $defaultBrand = null
    ) {
    }

    public function getTaggedFolders(string $tag, ?string $brand = null): array
    {
        return array_merge($this->findBrandedFolders($tag, $brand), $this->findCommonFolders($tag));
    }

    public function getAllTaggedFolders(string $tag): array
    {
        $paths = [];

        foreach ($this->brands as $brand) {
            foreach ($this->findBrandedFolders($tag, $brand) as $folder) {
                $paths[] = $folder;
            }
        }

        return array_merge($paths, $this->findCommonFolders($tag));
    }

    /**
     * @throws InvalidArgumentException If no provided brand or no default brand set.
     */
    protected function findBrandedFolders(string $tag, ?string $brand = null): array
    {
        $paths = [];

        $brand ??= $this->defaultBrand;

        if (!$brand) {
            throw new InvalidArgumentException('No provided brand and no default brand set.');
        }

        foreach ($this->folders as $folder) {
            if (in_array($tag, $folder['tags'])) {
                $paths[] = str_replace('{brand}', $brand, $folder['path']);
            }
        }

        return $paths;
    }

    protected function findCommonFolders(string $tag): array
    {
        $paths = [];

        foreach ($this->folders as $folder) {
            if (in_array($tag, $folder['tags']) && $folder['commonFolder']) {
                $paths[] = str_replace('{brand}', $folder['commonFolder'], $folder['path']);
            }
        }

        return $paths;
    }
}
