<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Command;

use AlloCine\BrandBundle\Model\BrandRepository;
use RuntimeException;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Process\Process;

#[AsCommand(
    name: 'app:cache:clear-all',
    description: 'Clears the cache for every brands'
)]
class CacheClearAllCommand extends Command
{
    public function __construct(
        protected readonly string $projectDir,
        private readonly BrandRepository $brandRepository,
        private readonly KernelInterface $kernel
    ) {
        parent::__construct();
    }

    protected function configure()
    {
        $this
            ->setDefinition([
                new InputOption('no-warmup', '', InputOption::VALUE_NONE, 'Do not warm up the cache'),
                new InputOption('no-optional-warmers', '', InputOption::VALUE_NONE, 'Skip optional cache warmers (faster)'),
            ])
            ->setHelp(
                <<<'TXT'
                The <info>%command.name%</info> command clears the application cache for a given environment
                and debug mode:
                
                  <info>php %command.full_name% --env=dev</info>
                  <info>php %command.full_name% --env=prod --no-debug</info>
                TXT
            );
    }

    public function __invoke(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $brands = $this->brandRepository->getAvailableBrands();
        $consoleExecutable = $this->projectDir . '/bin/console';

        foreach ($brands as $brand) {
            $io->title(sprintf('Clearing the cache for the <info>%s</info> brand', $brand));

            $command = [
                $consoleExecutable,
                'cache:clear',
                '--env',
                $this->kernel->getEnvironment(),
                '--brand',
                $brand
            ];

            if (true === $input->getOption('no-warmup')) {
                $command[] = '--no-warmup';
            }

            if (true === $input->getOption('no-optional-warmers')) {
                $command[] = '--no-optional-warmers';
            }

            $process = new Process($command);
            $process->setTimeout(null);

            $process->run(function ($type, $text) use ($output): void {
                $output->write($text);
            });

            if (!$process->isSuccessful()) {
                throw new RuntimeException($process->getErrorOutput());
            }
            $process = null;
            $command = null;
        }

        return Command::SUCCESS;
    }
}
