<?php

namespace Tests\BrandBundle\Tests\Routing\Builtin;

use AlloCine\BrandBundle\Model\CacheControl;
use AlloCine\BrandBundle\Routing\Builtin\CacheExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\FilterResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

class CacheExtensionTest extends TestCase
{
    /**
     * @expectedException \LogicException
     * @expectedExceptionMessage Invalid cache.enabled value "bar". Must be false or true.
     */
    public function testInvalidEnabledConfig()
    {
        $extension = new CacheExtension(3600);

        $extension->validate([ 'enabled' => 'bar' ]);
    }

    /**
     * @expectedException \LogicException
     * @expectedExceptionMessage Invalid cache.ttl value "-1000". Must be valid positive integer.
     */
    public function testInvalidTtlConfig()
    {
        $extension = new CacheExtension(3600);

        $extension->validate([ 'ttl' => -1000 ]);
    }

    /**
     * @expectedException \LogicException
     * @expectedExceptionMessage Invalid cache short notation value "bar". Must be false or true.
     */
    public function testInvalidShortConfig()
    {
        $extension = new CacheExtension(3600);

        $extension->validate('bar');
    }


    public function testPostLoad()
    {
        $collection = new RouteCollection();
        $collection->add('1', new Route('/1'));
        $collection->add('2', new Route('/2', [], [], [ 'cache' => true ]));
        $collection->add('3', new Route('/3', [], [], [ 'cache' => false ]));
        $collection->add('4', new Route('/4', [], [], [ 'cache' => [ 'enabled' => true ] ]));
        $collection->add('5', new Route('/5', [], [], [ 'cache' => [ 'enabled' => false ] ]));
        $collection->add('6', new Route('/6', [], [], [ 'cache' => [ 'ttl' => 1000 ] ]));

        $extension = new CacheExtension(3600);
        $extension->postLoad($collection);

        $enabledCache = (new CacheControl())->setEnabled(true)->setTtl(3600);
        $disabledCache = (new CacheControl())->setEnabled(false)->setTtl(3600);
        $this->assertEquals($enabledCache, $collection->get('1')->getDefault('_cache_control'), 'route 1');
        $this->assertEquals($enabledCache, $collection->get('2')->getDefault('_cache_control'), 'route 2');
        $this->assertEquals($disabledCache, $collection->get('3')->getDefault('_cache_control'), 'route 3');
        $this->assertEquals($enabledCache, $collection->get('4')->getDefault('_cache_control'), 'route 4');
        $this->assertEquals($disabledCache, $collection->get('5')->getDefault('_cache_control'), 'route 5');
        $this->assertEquals((new CacheControl())->setEnabled(true)->setTtl(1000), $collection->get('6')->getDefault('_cache_control'), 'route 6');
    }

    public function testCacheResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response();
        $request = new Request();
        $request->attributes->set('_cache_control', (new CacheControl())->setEnabled(true)->setTtl(1000));
        $event = new FilterResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertSame('1000', $event->getResponse()->headers->getCacheControlDirective('s-maxage'));
    }

    public function testNoCacheResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response();
        $request = new Request();
        $request->attributes->set('_cache_control', (new CacheControl())->setEnabled(false)->setTtl(1000));
        $event = new FilterResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertFalse($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('no-store'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('private'));
    }

    public function testErrorResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response('', 500);
        $request = new Request();
        $request->attributes->set('_cache_control', (new CacheControl())->setEnabled(true)->setTtl(1000));
        $event = new FilterResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertFalse($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('no-cache'));
    }

    private function getKernel()
    {
        return $this
            ->getMockBuilder('Symfony\Component\HttpKernel\HttpKernelInterface')
            ->disableOriginalConstructor()
            ->getMock()
        ;
    }
}
