<?php

namespace Tests\BrandBundle\Tests\Routing\Builtin;

use AlloCine\BrandBundle\Routing\Builtin\CacheExtension;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\HttpKernelInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

class CacheExtensionTest extends TestCase
{
    public function testInvalidEnabledConfig()
    {
        $this->expectException(\LogicException::class);
        $this->expectExceptionMessage('Invalid cache.enabled value "bar". Must be false or true.');

        $extension = new CacheExtension(3600);

        $extension->validate([ 'enabled' => 'bar' ]);
    }

    public function testInvalidTtlConfig()
    {
        $this->expectException(\LogicException::class);
        $this->expectExceptionMessage('Invalid cache.ttl value "-1000". Must be valid positive integer.');

        $extension = new CacheExtension(3600);

        $extension->validate([ 'ttl' => -1000 ]);
    }

    public function testInvalidShortConfig()
    {
        $this->expectException(\LogicException::class);
        $this->expectExceptionMessage('Invalid cache short notation value "bar". Must be false or true.');

        $extension = new CacheExtension(3600);

        $extension->validate('bar');
    }


    public function testPostLoad()
    {
        $collection = new RouteCollection();
        $collection->add('1', new Route('/1'));
        $collection->add('2', new Route('/2', [], [], [ 'cache' => true ]));
        $collection->add('3', new Route('/3', [], [], [ 'cache' => false ]));
        $collection->add('4', new Route('/4', [], [], [ 'cache' => [ 'enabled' => true ] ]));
        $collection->add('5', new Route('/5', [], [], [ 'cache' => [ 'enabled' => false ] ]));
        $collection->add('6', new Route('/6', [], [], [ 'cache' => [ 'ttl' => 1000 ] ]));

        $extension = new CacheExtension(3600);
        $extension->postLoad($collection);

        $this->assertSame(['enabled' => true, 'ttl' => 3600], $collection->get('1')->getDefault('_cache_control'));
        $this->assertSame(['enabled' => true, 'ttl' => 3600], $collection->get('2')->getDefault('_cache_control'));
        $this->assertSame(['enabled' => false, 'ttl' => 3600], $collection->get('3')->getDefault('_cache_control'));
        $this->assertSame(['enabled' => true, 'ttl' => 3600], $collection->get('4')->getDefault('_cache_control'));
        $this->assertSame(['enabled' => false, 'ttl' => 3600], $collection->get('5')->getDefault('_cache_control'));
        $this->assertSame(['enabled' => true, 'ttl' => 1000], $collection->get('6')->getDefault('_cache_control'));
    }

    public function testCacheResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response();
        $request = new Request();
        $request->attributes->set('_cache_control', ['enabled' => true, 'ttl' => 1000]);
        $event = new ResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertSame('1000', $event->getResponse()->headers->getCacheControlDirective('s-maxage'));
    }

    public function testNoCacheResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response();
        $request = new Request();
        $request->attributes->set('_cache_control', ['enabled' => false, 'ttl' => 1000]);
        $event = new ResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertFalse($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('no-store'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('private'));
    }

    public function testErrorResponse()
    {
        $extension = new CacheExtension(3600);

        $response = new Response('', 500);
        $request = new Request();
        $request->attributes->set('_cache_control', ['enabled' => true, 'ttl' => 1000]);
        $event = new ResponseEvent($this->getKernel(), $request, HttpKernelInterface::MASTER_REQUEST, $response);

        $extension->onKernelResponse($event);

        $this->assertFalse($event->getResponse()->headers->hasCacheControlDirective('public'));
        $this->assertTrue($event->getResponse()->headers->hasCacheControlDirective('no-cache'));
    }

    private function getKernel()
    {
        return $this
            ->getMockBuilder(HttpKernelInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;
    }
}
