<?php

namespace AlloCine\BrandBundle\Command;

use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Process\Process;

/**
 * Description of CacheClearAllCommand
 */
class CacheClearAllCommand extends Command
{
    use ContainerAwareTrait;

    /**
     * @var string
     */
    protected $projectDir;

    /**
     * Command constructor.
     *
     * @param string      $projectDir
     * @param string|null $name
     */
    public function __construct(string $projectDir, string $name = null)
    {
        parent::__construct($name);

        $this->projectDir = $projectDir;
    }

    /**
     * @required
     * @param ContainerInterface $container
     */
    public function setContainer(ContainerInterface $container = null)
    {
        $this->container = $container;
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('cache:clear:all')
            ->setDefinition([
                new InputOption('no-warmup', '', InputOption::VALUE_NONE, 'Do not warm up the cache'),
                new InputOption('no-optional-warmers', '', InputOption::VALUE_NONE, 'Skip optional cache warmers (faster)'),
            ])
            ->setDescription('Clears the cache for every brands')
            ->setHelp(<<<EOF
The <info>%command.name%</info> command clears the application cache for a given environment
and debug mode:

  <info>php %command.full_name% --env=dev</info>
  <info>php %command.full_name% --env=prod --no-debug</info>
EOF);
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $brands = $this->container->get('brand.brand_repository')->getAvailableBrands();
        $consoleExecutable = $this->projectDir . '/bin/console';

        foreach ($brands as $brand) {
            $io->title(sprintf('Clearing the cache for the <info>%s</info> brand', $brand));

            $command = [
                $consoleExecutable,
                'cache:clear',
                '--env',
                $this->container->get('kernel')->getEnvironment(),
                '--brand',
                $brand
            ];

            if (true === $input->getOption('no-warmup')) {
                $command[] = '--no-warmup';
            }

            if (true === $input->getOption('no-optional-warmers')) {
                $command[] = '--no-optional-warmers';
            }

            $process = new Process($command);
            $process->setTimeout(null);

            $process->run(function ($type, $text) use ($output): void {
                $output->write($text);
            });

            if (!$process->isSuccessful()) {
                throw new \RuntimeException($process->getErrorOutput());
            }
            $process = null;
            $command = null;
        }

        return 0;
    }
}
