<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Tests\Kernel;

use LogicException;
use AlloCine\BrandBundle\Tests\TestKernel;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Config\Loader\LoaderInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;

final class BrandableKernelTest extends TestCase
{
    public function testDirectories(): void
    {
        $kernel = new TestKernel('dev', false, 'brand1');

        $this->assertSame($kernel->getProjectDir().'/var/cache/dev/brand1', $kernel->getCacheDir());
        $this->assertSame($kernel->getProjectDir().'/var/log', $kernel->getLogDir());
    }

    public function testSetBrand(): void
    {
        $kernel = new TestKernel('dev', false, 'brand1');
        $kernel->setBrand('brand2');

        $this->assertSame('brand2', $kernel->getBrand());
    }

    public function testName(): void
    {
        $kernel = new TestKernel('dev', false, 'brand1');

        $this->assertSame('AlloCine_BrandBundle_Tests_TestKernelDevContainer', $kernel->getName());
    }

    public function testNameWithDebug(): void
    {
        $kernel = new TestKernel('dev', true, 'brand1');

        $this->assertSame('AlloCine_BrandBundle_Tests_TestKernelDevDebugContainer', $kernel->getName());
    }

    public function testSetBrandWhileBooted(): void
    {
        $this->expectException(LogicException::class);
        $this->expectExceptionMessage('Brand can only be changed when kernel is not booted');

        $kernel = new TestKernel('dev', true, 'brand1');
        $kernel->boot();
        $kernel->setBrand('brand2');
    }

    public function testConfigureContainerWithNoEnvForBrand(): void
    {
        $kernel = new TestKernel('dev', false, 'brand1');

        $container = $this
            ->getMockBuilder(ContainerBuilder::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader = $this
            ->getMockBuilder(LoaderInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader
            ->expects($this->exactly(2))
            ->method('load')
            ->willReturnCallback( fn(): array => [
                $kernel->getProjectDir().'/config/config_dev'.TestKernel::CONFIG_EXTS,
                $kernel->getProjectDir().'/config/brand1/config'.TestKernel::CONFIG_EXTS
            ]
            )
        ;

        $kernel->configureContainer($container, $loader);
    }

    public function testConfigureContainerWithEnvForBrand(): void
    {
        $kernel = new TestKernel('dev', true, 'brand2');

        $container = $this
            ->getMockBuilder(ContainerBuilder::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader = $this
            ->getMockBuilder(LoaderInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader
            ->expects($this->exactly(2))
            ->method('load')
            ->willReturnCallback(fn(): array => [
                $kernel->getProjectDir().'/config/config_dev'.TestKernel::CONFIG_EXTS,
                $kernel->getProjectDir().'/config/brand2/config_dev'.TestKernel::CONFIG_EXTS,
            ])
        ;

        $kernel->configureContainer($container, $loader);
    }

    public function testRegisterContainerConfigurationWithNoConfigForBrand(): void
    {
        $kernel = new TestKernel('dev', true, 'brand3');

        $container = $this
            ->getMockBuilder(ContainerBuilder::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader = $this
            ->getMockBuilder(LoaderInterface::class)
            ->disableOriginalConstructor()
            ->getMock()
        ;

        $loader
            ->expects($this->once())
            ->method('load')
            ->with($kernel->getProjectDir().'/config/config_dev'.TestKernel::CONFIG_EXTS)
        ;

        $kernel->configureContainer($container, $loader);
    }

    public function testSerialize(): void
    {
        $kernel = new TestKernel('dev', false, 'brand42');

        $clone = unserialize(serialize($kernel));

        self::assertSame($kernel->getBrand(), $clone->getBrand());
    }

}
