<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Kernel;

use Override;
use LogicException;
use Exception;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\HttpKernel\Kernel;
use Symfony\Component\Config\Loader\LoaderInterface;

abstract class BrandableKernel extends Kernel
{
    public const string CONFIG_EXTS = '.{php,xml,yaml,yml}';

    public function __construct(string $environment, bool $debug, private string $brand = 'allocine')
    {
        parent::__construct($environment, $debug);
    }

    #[Override]
    public function getCacheDir(): string
    {
        return $this->getProjectDir().'/var/cache/'.$this->getEnvironment().'/'.$this->brand;
    }

    private function getConfigDir(): string
    {
        return $this->getProjectDir().'/config';
    }

    /**
     * @throws Exception
     */
    protected function configureContainer(ContainerBuilder $container, LoaderInterface $loader): void
    {
        $envConfig = $this->getConfigDir().'/config_'.$this->getEnvironment(). self::CONFIG_EXTS;
        if (!in_array(glob($envConfig, GLOB_BRACE), [[], false], true)) {
            $loader->load($envConfig, 'glob');
        }

        $brandEnvConfig = $this->getConfigDir().'/'.$this->brand.'/config_'.$this->getEnvironment().self::CONFIG_EXTS;
        $brandConfig = $this->getConfigDir().'/'.$this->brand.'/config' . self::CONFIG_EXTS;
        if (!in_array(glob($brandEnvConfig, GLOB_BRACE), [[], false], true)) {
            $loader->load($brandEnvConfig, 'glob');
        } elseif (!in_array(glob($brandConfig, GLOB_BRACE), [[], false], true)) {
            $loader->load($brandConfig, 'glob');
        }
    }

    public function getBrand(): string
    {
        return $this->brand;
    }

    /**
     * @throws LogicException
     */
    public function setBrand(string $brand): void
    {
        if ($this->booted) {
            throw new LogicException('Brand can only be changed when kernel is not booted');
        }

        $this->brand = $brand;
    }

    public function getName(): string
    {
        return $this->getContainerClass();
    }

    #[Override]
    protected function getKernelParameters(): array
    {
        return array_merge(parent::getKernelParameters(), [
            'brand' => $this->brand
        ]);
    }

    public function __serialize(): array
    {
        return [$this->environment, $this->debug, $this->brand];
    }

    public function __unserialize(array $data): void
    {
        [$environment, $debug, $brand] = $data;

        $this->__construct($environment, $debug, $brand);
    }
}
