<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Routing\Builtin;

use LogicException;
use AlloCine\BrandBundle\Routing\Extension\LoaderExtensionInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\Routing\RouteCollection;

class CacheExtension implements LoaderExtensionInterface, EventSubscriberInterface
{
    public function __construct(private readonly int $defaultTtl)
    {
    }

    public function getName(): string
    {
        return 'cache';
    }

    public static function getSubscribedEvents(): array
    {
        return [
            KernelEvents::RESPONSE => 'onKernelResponse',
        ];
    }

    public function validate($config): void
    {
        if (is_null($config)) {
            return;
        }

        if (is_array($config)) {
            if (isset($config['enabled']) && !is_bool($config['enabled'])) {
                throw new LogicException(sprintf('Invalid cache.enabled value "%s". Must be false or true.', $config['enabled']));
            }
            if (isset($config['ttl']) && (!is_int($config['ttl']) || $config['ttl'] < 0)) {
                throw new LogicException(sprintf('Invalid cache.ttl value "%s". Must be valid positive integer.', $config['ttl']));
            }
        } elseif (!is_bool($config)) {
            throw new LogicException(sprintf('Invalid cache short notation value "%s". Must be false or true.', $config));
        }
    }

    public function postLoad(RouteCollection $collection): void
    {
        $defaults = [ 'enabled' => true, 'ttl' => $this->defaultTtl ];

        foreach ($collection->all() as $route) {
            $cache = $route->getOption('cache');

            if (is_null($cache)) {
                $cache = $defaults;
            } elseif (is_bool($cache)) {
                $cache = [ 'enabled' => $cache ];
            }

            $route->setDefault('_cache_control', array_merge($defaults, $cache));
        }
    }

    public function onKernelResponse(ResponseEvent $event): void
    {
        $response = $event->getResponse();

        if (!$response->isSuccessful()) {
            return;
        }

        $options = $event->getRequest()->get('_cache_control');

        if (!isset($options['enabled']) || !$options['enabled']) {
            $event->setResponse($this->disableCache($response));

            return;
        }

        $event->setResponse($this->enableCache($response, $options['ttl']));
    }

    protected function enableCache(Response $response, ?int $ttl = null): Response
    {
        $ttl = $ttl ?: $this->defaultTtl;

        $response->setPublic()->setMaxAge($ttl);
        $response->setPublic()->setSharedMaxAge($ttl);
        // To be sure we never have "no-store" in headers (for example, when forwarding controller)
        $response->headers->removeCacheControlDirective('no-store');

        return $response;
    }

    protected function disableCache(Response $response): Response
    {
        $response->setPrivate();
        $response->headers->addCacheControlDirective('no-store');

        return $response;
    }
}
