
# Advanced usage

## Custom config

You will certainly need to change configuration based on brands. For instance, each brand could have its own twig globals.

When the BrandableKernel boots it tries to read extra config in the following order :

- `app/config/{brand}/config_{env}.yml`
- `app/config/{brand}/config.yml`

And then if any of this file is found, it will be merged on top of the Symfony standard config file.

This allows you to override some config based on the brand, let's create `app/config/fishstore/config.yml` :

```
twig:
    globals:
        default_title: 'FishStore - Get that fish you need !'
```

And that's it ! Twig globals are overriden for fishstore.

But what if you want to add some config for a brand that other brands can see ? There come the `extras`. You can add
any extra variable you want in the configuration of the bundle :

```yml
allo_cine_brand:
    brands:
        fishstore:
            host: www.fishstore.dev
            locale: en_US
            extras:
                redis_queue: fish
        boutiquepoisson:
            host: www.boutiquepoisson.dev
            locale: fr_FR
                redis_queue: poisson
```

This is now accessible via the `brand.brand_repository` service :

```php
foreach ($container->get('brand.brand_repository')->all() as $brand => $config) {
    $redisQueue = $config['extras']['redis_queue'];

    // [...]
}
```

Take a look at `BrandRepository` to see what else you can get at runtime about brands.

## More with folders

The folder entry of the configuration of this bundle allows to know where the twig files live. But there is more !

For instance, you can define a fallback in case a view does not exist for a brand :

```yml
brands:
    # [...] brands declaration
    folders:
        views:
            path: "%kernel.root_dir%/Resources/views/{brand}"
            tags: ['twig']
            commonFolder: 'default'
```

With this configuration, using the fishstore brand, if the file `home.html.twig` is not found in `Resources/views/fishstore`
it will be searched in `Resources/views/default`.

But wait ! There is more !

You can also add more view folders :

```yml
brands:
    # [...] brands declaration
    folders:
        views_main:
            path: "%kernel.root_dir%/Resources/views/{brand}"
            tags: ['twig']
            commonFolder: 'default'
        views_alternative:
            path: "%kernel.root_dir%/otherwere/views/{brand}"
            tags: ['twig']
            commonFolder: 'default'
```

With this configuration, `home.html.twig` for fishstore would be resolved as follow :

- `Resources/views/fishstore/home.html.twig`
- `otherwere/views/fishstore/home.html.twig`
- `Resources/views/default/home.html.twig`
- `otherwere/views/default/home.html.twig`

There is still more !

You can declare any folder you want in the `folders` entry and the service `brand.folder_repository` will allow you to fetch
them at runtime :

```yaml
brands:
    # [...] brands declaration
    folders:
        sql:
            path: "%kernel.root_dir%/Resources/sql/{brand}"
            tags: ['sql']
            commonFolder: 'main'
```

And now, using the repository :

```php
$container->get('brand.folder_repository')->getTaggedFolders('sql');
```

Would return `['Resources/sql/fishstore', 'Resources/sql/main']`.

You can also get this folders for all brands at once :

```php
$container->get('brand.folder_repository')->getAllTaggedFolders('sql');
```

This would return : `['Resources/sql/fishstore', 'Resources/sql/boutiquepoisson', 'Resources/sql/main']`

## Bindings

Bindings are a way to integrate with other bundles.

### AllocineI18nBundle

```yaml
    bindings:
        allocineI18n:
            sourceFolders: ['twig']
```

If the AllocineI18nBundle is enabled, this will add every folder tagged with `twig` in the BrandBundle to the
list of folders to scan for extracting translations.

It will also add each brand's locale to the list of locales to generate.

See [the AllocineI18nBundle README](https://github.com/allocine/i18n-bundle).

## Routing extensions

The bundle also provides optional routing features that can be enabled using configuration.

### Automatic cache headers

```yaml
allo_cine_brand:
    # [...]
    builtins:
        cache:
            enabled: true
            default_ttl: 3600
```

With this configuration, every response from any route will by default have the `Cache-Control: public, s-maxage=3600`
header.

You can tune this on a per route basis :

```yaml
myroute:
    path: /foo
    options:
        cache:
            ttl: 86400

myroute2:
    path: /bar
    options:
        cache:
            enabled: false
```

There is also a short notation for disabling cache that is equivalent to the one used for `route2` :

```yaml
myroute3:
    path: /baz
    options:
        cache: false
```

## Custom routing

The BrandBundle completely overrides the yaml loader of Symfony. In some cases you might want to add some custom features
to your routing such as custom configuration.

This is a complex but powerful feature. To see what you can do, look at `GeneratorExtensionInterface`
and `LoaderExtensionInterface`.
