<?php

namespace AlloCine\BrandBundle\Routing\Builtin;

use AlloCine\BrandBundle\Routing\Extension\LoaderExtensionInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\Routing\RouteCollection;

class CacheExtension implements LoaderExtensionInterface
{
    /**
     * @var int
     */
    private $defaultTtl;

    /**
     * @param int $defaultTtl
     */
    public function __construct($defaultTtl)
    {
        $this->defaultTtl = $defaultTtl;
    }

    /**
     * {@inheritDoc}
     */
    public function getName()
    {
        return 'cache';
    }

    /**
     * {@inheritDoc}
     */
    public function validate($cache)
    {
        if (is_null($cache)) {
            return;
        }

        if (is_array($cache)) {
            if (isset($cache['enabled']) && !is_bool($cache['enabled'])) {
                throw new \LogicException(sprintf('Invalid cache.enabled value "%s". Must be false or true.', $cache['enabled']));
            }
            if (isset($cache['ttl']) && (!is_integer($cache['ttl']) || $cache['ttl'] < 0)) {
                throw new \LogicException(sprintf('Invalid cache.ttl value "%s". Must be valid positive integer.', $cache['ttl']));
            }
        } elseif (!is_bool($cache)) {
            throw new \LogicException(sprintf('Invalid cache short notation value "%s". Must be false or true.', $cache));
        }
    }

    /**
     * {@inheritDoc}
     */
    public function postLoad(RouteCollection $collection)
    {
        $defaults = [ 'enabled' => true, 'ttl' => $this->defaultTtl ];

        foreach ($collection->all() as $route) {
            $cache = $route->getOption('cache');

            if (is_null($cache)) {
                $cache = $defaults;
            } elseif (is_bool($cache)) {
                $cache = [ 'enabled' => $cache ];
            }

            $route->setDefault('_cache_control', array_merge($defaults, $cache));
        }
    }

    /**
     * @param ResponseEvent $event
     */
    public function onKernelResponse(ResponseEvent $event)
    {
        $response = $event->getResponse();

        if (!$response->isSuccessful()) {
            return;
        }

        $options  = $event->getRequest()->get('_cache_control');

        if (!isset($options['enabled']) || !$options['enabled']) {
            return $event->setResponse($this->disableCache($response));
        }

        return $event->setResponse($this->enableCache($response, $options['ttl']));
    }

    /**
     * @param Response $response
     * @param int      $ttl
     *
     * @return Response
     */
    protected function enableCache(Response $response, $ttl = null)
    {
        $ttl = $ttl ? $ttl : $this->defaultTtl;

        $response->setPublic()->setMaxAge($ttl);
        $response->setPublic()->setSharedMaxAge($ttl);
        // To be sure we never have "no-store" in headers (for example, when forwarding controller)
        $response->headers->removeCacheControlDirective('no-store');

        return $response;
    }

    /**
     * @param Response $response
     *
     * @return Response
     */
    protected function disableCache(Response $response)
    {
        $response->setPrivate();
        $response->headers->addCacheControlDirective('no-store');

        return $response;
    }
}
