<?php

namespace AlloCine\BrandBundle\Routing;

use AlloCine\BrandBundle\Routing\Extension\GeneratorExtensionInterface;
use Psr\Log\LoggerInterface;
use Symfony\Bundle\FrameworkBundle\Routing\Router;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Routing\Exception\ResourceNotFoundException;
use Symfony\Component\Routing\Exception\RouteNotFoundException;
use Symfony\Component\Routing\RequestContext;

/**
 * Class BrandRouter
 *
 * This class overrides the router's generate method, by injecting
 * the default brand, or a brand which is exposed as a route parameter.
 * Overriding the default brand can be done using the '_brand' parameter.
 */
class BrandRouter extends Router
{
    public function __construct(
        ContainerInterface $container,
        $resource,
        array $options = [],
        RequestContext $context = null,
        LoggerInterface $logger = null
    ) {
        $this->logger = $logger;
        parent::__construct($container, $resource, $options, $context);
    }

    /**
     * @var string
     */
    private $defaultBrand;

    /**
     * @var GeneratorExtensionInterface[]
     */
    private $extensions = [];

    /**
     * @param GeneratorExtensionInterface $extension
     */
    public function addExtension(GeneratorExtensionInterface $extension)
    {
        $this->extensions[] = $extension;
    }

    /**
     * @param string $brand
     */
    public function setDefaultBrand($brand)
    {
        $this->defaultBrand = $brand;
    }

    /**
     * @inheritdoc
     */
    public function generate($name, $parameters = [], $referenceType = self::ABSOLUTE_PATH)
    {
        if (isset($parameters['_brand'])) {
            $brand = $parameters['_brand'];
            unset($parameters['_brand']);
        } else {
            $brand = $this->defaultBrand;
        }

        foreach ($this->extensions as $extension) {
            $name       = $extension->overrideName($name, $parameters);
            $parameters = $extension->overrideParameters($name, $parameters);
        }

        if ($brand) {
            try {
                return parent::generate($brand.'__'.$name, $parameters, $referenceType);
            } catch (RouteNotFoundException $e) {
                // proceed.
            }
        }

        try {
            return parent::generate($name, $parameters, $referenceType);
        } catch (RouteNotFoundException $e) {
            throw new ResourceNotFoundException(sprintf('Unable to generate a URL for the named route "%s" as such route does not exist for brand "%s"', $name, $brand));
        }
    }
}
