# What is AllocineBrandBundle ?

The main goal of this bundle is to allow a website to be displayed with different views, routes and domains.

Here at Allociné we call this "brands", hence the name of the bundle.

# How to install it ?

Add this to your composer.json :

```json
{
    "require": {
        "allocine/brand-bundle": "1.0.*"
    }
}
```

Now you need not only to add the bundle to your kernel, but also change the kernel a little bit by
extending `BrandableKernel` and letting it handle `getCacheDir`, `getLogDir` and `registerContainerConfiguration`.
Here is a simple kernel that will do the job :


```php
namespace App;

use AlloCine\BrandBundle\Kernel\BrandableKernel;
use Symfony\Bundle\FrameworkBundle\Kernel\MicroKernelTrait;
use Symfony\Component\Config\Loader\LoaderInterface;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\Routing\RouteCollectionBuilder;

class Kernel extends BrandableKernel
{
    use MicroKernelTrait;

    /**
     * @return string
     */
    public function getProjectDir()
    {
        return realpath(__DIR__.'/../');
    }

    /**
     * @return string
     */
    public function getLogDir()
    {
        // sf 4.0 default log folder
        return $this->getProjectDir().'/var/log';
    }

    public function getConfigDir()
    {
        // sf 4.0 default config folder
        return $this->getProjectDir().'/config';
    }

    public function registerBundles()
    {
        $contents = require $this->getProjectDir().'/config/bundles.php';
        foreach ($contents as $class => $envs) {
            if (isset($envs['all']) || isset($envs[$this->environment])) {
                yield new $class();
            }
        }
    }

    /**
     * @param ContainerBuilder $container
     * @param LoaderInterface  $loader
     *
     * @throws \Exception
     */
    protected function configureContainer(ContainerBuilder $container, LoaderInterface $loader)
    {
        parent::configureContainer($container, $loader);

        $container->setParameter('container.autowiring.strict_mode', true);
        $container->setParameter('container.dumper.inline_class_loader', true);
        $confDir = $this->getProjectDir().'/config';
        $loader->load($confDir.'/packages/*'.self::CONFIG_EXTS, 'glob');
        if (is_dir($confDir.'/packages/'.$this->environment)) {
            $loader->load($confDir.'/packages/'.$this->environment.'/**/*'.self::CONFIG_EXTS, 'glob');
        }
        $loader->load($confDir.'/services'.self::CONFIG_EXTS, 'glob');
        $loader->load($confDir.'/services_'.$this->environment.self::CONFIG_EXTS, 'glob');
    }

    /**
     * @param RouteCollectionBuilder $routes
     *
     * @throws \Symfony\Component\Config\Exception\FileLoaderLoadException
     */
    protected function configureRoutes(RouteCollectionBuilder $routes)
    {
        $confDir = $this->getProjectDir().'/config';
        if (is_dir($confDir.'/routes/')) {
            $routes->import($confDir.'/routes/*'.self::CONFIG_EXTS, '/', 'glob');
        }
        if (is_dir($confDir.'/routes/'.$this->environment)) {
            $routes->import($confDir.'/routes/'.$this->environment.'/**/*'.self::CONFIG_EXTS, '/', 'glob');
        }
        $routes->import($confDir.'/routes'.self::CONFIG_EXTS, '/', 'glob');
    }
}

```

Finally, let's change the console entry point (`bin/console`) to handle brands :

```php

#!/usr/bin/env php
<?php

use App\Kernel;
use AlloCine\BrandBundle\Console\Application;
use Symfony\Component\Console\Input\ArgvInput;
use Symfony\Component\Debug\Debug;
use Symfony\Component\Dotenv\Dotenv;

set_time_limit(0);

require __DIR__.'/../vendor/autoload.php';

if (!class_exists(Application::class)) {
    throw new \RuntimeException('You need to add "allocine/brand-bundle" as a Composer dependency.');
}

if (!isset($_SERVER['APP_ENV'])) {
    (new Dotenv())->load(__DIR__.'/../.env');
}

$input = new ArgvInput();
$env = $input->getParameterOption(['--env', '-e'], $_SERVER['APP_ENV'] ?? 'dev');
$brand = $input->getParameterOption(array('--brand', '-b'), getenv('SF_BRAND') ?: 'allocine');

$debug = ($_SERVER['APP_DEBUG'] ?? ('prod' !== $env)) && !$input->hasParameterOption(['--no-debug', '']);

if ($debug) {
    umask(0000);

    if (class_exists(Debug::class)) {
        Debug::enable();
    }
}

$kernel = new Kernel($env, $debug, $brand);
$application = new Application($kernel);
$application->run($input);
```

You are now ready to go !

# Basic Usage

The first thing you need to do is declaring the brands as the bundle will refuse to use any non declared brands.
Let's say we have many websites for selling fishes.

```yml
allo_cine_brand:
    brands:
        fishstore:
            host: www.fishstore.dev # Used for cross-brands links
            locale: en_US # The default locale
        boutiquepoisson:
            host: www.boutiquepoisson.dev
            locale: fr_FR
```

But that's not enough, each brand has it's own views and urls, so let's declare a route for our shop cart :

```yml
cart:
    path:
        fishstore: /cart/view
        boutiquepoisson: /mon-panier
    defaults:
        _controller: AppBundle:Cart:index
```

Ok, so now, every time you visit `www.fishstore.dev/cart/view` you reach the right controller. But how to handle
different views ?

Just by declaring where to find them :

```yml
brands:
    # [...] brands declaration as seen above
    folders:
        views:
            path: "%kernel.root_dir%/Resources/views/{brand}"
            tags: ['twig'] # This is important as it tells the bundle it is a view folder.
```

With this setting, every time a call to the `render` method of Twig, the template are searched in the specified folder.
For instance, with fishstore : `render('home.html.twig')` will resolve to the file `%kernel.root_dir%/Resources/views/fishstore`.

You are now ready to multi-brand ! Take a look at the [advanced usages](Resources/doc/advanced.md).

# FAQ
## How do I disable a route for a brand ?

Sometime a route is available for a brand and not for the other, simply omit it in the routing :

```yml
cart:
    path:
        fishstore: /cart/view
        # boutiquepoisson: /mon-panier <- Will throw a 404 for this brand.
    defaults:
        _controller: AppBundle:Cart:index
```
