<?php

namespace AlloCine\BrandBundle\Kernel;

use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\HttpKernel\Kernel;
use Symfony\Component\Config\Loader\LoaderInterface;

abstract class BrandableKernel extends Kernel implements \Serializable
{
    const CONFIG_EXTS = '.{php,xml,yaml,yml}';

    /**
     * @var string
     */
    private $brand;

    /**
     * @param string $environment
     * @param bool $debug
     * @param string $brand
     */
    public function __construct(string $environment, bool $debug, string $brand = 'allocine')
    {
        $this->brand = $brand;

        parent::__construct($environment, $debug);
    }

    /**
     * @return string
     */
    public function getCacheDir(): string
    {
        return $this->getProjectDir().'/var/cache/'.$this->getEnvironment().'/'.$this->brand;
    }

    /**
     * @return string
     */
    public function getLogDir(): string
    {
        return $this->getProjectDir().'/var/log';
    }

    /**
     * @return string
     */
    public function getConfigDir(): string
    {
        return $this->getProjectDir().'/config';
    }

    /**
     * @param ContainerBuilder $container
     * @param LoaderInterface  $loader
     *
     * @throws \Exception
     */
    protected function configureContainer(ContainerBuilder $container, LoaderInterface $loader)
    {
        $envconfig = $this->getConfigDir().'/config_'.$this->getEnvironment(). self::CONFIG_EXTS;
        if (!empty(glob($envconfig, GLOB_BRACE))) {
            $loader->load($envconfig, 'glob');
        }

        $brandEnvConfig = $this->getConfigDir().'/'.$this->brand.'/config_'.$this->getEnvironment().self::CONFIG_EXTS;
        $brandConfig = $this->getConfigDir().'/'.$this->brand.'/config' . self::CONFIG_EXTS;
        if (!empty(glob($brandEnvConfig, GLOB_BRACE))) {
            $loader->load($brandEnvConfig, 'glob');
        } elseif (!empty(glob($brandConfig, GLOB_BRACE))) {
            $loader->load($brandConfig, 'glob');
        }
    }

    /**
     * @return string
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param string $brand
     *
     * @throws \LogicException
     */
    public function setBrand($brand)
    {
        if (true === $this->booted) {
            throw new \LogicException('Brand can only be changed when kernel is not booted');
        }

        $this->brand = $brand;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return ucfirst($this->brand) . ($this->isDebug() ? 'Debug' : '' );
    }

    /**
     * @return string
     */
    protected function getContainerClass()
    {
        return parent::getContainerClass().ucfirst($this->environment).'ProjectContainer';
    }

    public function serialize()
    {
        return serialize([$this->environment, $this->debug, $this->brand]);
    }

    public function unserialize($serialized)
    {
        list($environment, $debug, $brand) = unserialize($serialized, ['allowed_classes' => false]);

        $this->__construct($environment, $debug, $brand);
    }

    /**
     * {@inheritDoc}
     */
    protected function getKernelParameters()
    {
        return array_merge(parent::getKernelParameters(), [
            'brand' => $this->brand
        ]);
    }
}
