<?php

namespace AlloCine\BrandBundle\Routing\Builtin;

use AlloCine\BrandBundle\Model\CacheControl;
use AlloCine\BrandBundle\Routing\Extension\LoaderExtensionInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\FilterResponseEvent;
use Symfony\Component\Routing\RouteCollection;

class CacheExtension implements LoaderExtensionInterface
{
    /**
     * @var int
     */
    private $defaultTtl;

    /**
     * @param int $defaultTtl
     */
    public function __construct($defaultTtl)
    {
        $this->defaultTtl = $defaultTtl;
    }

    /**
     * {@inheritDoc}
     */
    public function getName()
    {
        return 'cache';
    }

    /**
     * {@inheritDoc}
     */
    public function validate($cache)
    {
        if (is_null($cache)) {
            return;
        }

        if (is_array($cache)) {
            if (isset($cache['enabled']) && !is_bool($cache['enabled'])) {
                throw new \LogicException(sprintf('Invalid cache.enabled value "%s". Must be false or true.', $cache['enabled']));
            }
            if (isset($cache['ttl']) && (!is_integer($cache['ttl']) || $cache['ttl'] < 0)) {
                throw new \LogicException(sprintf('Invalid cache.ttl value "%s". Must be valid positive integer.', $cache['ttl']));
            }
        } elseif (!is_bool($cache)) {
            throw new \LogicException(sprintf('Invalid cache short notation value "%s". Must be false or true.', $cache));
        }
    }

    /**
     * {@inheritDoc}
     */
    public function postLoad(RouteCollection $collection)
    {
        foreach ($collection->all() as $route) {
            $cache = (new CacheControl())->setEnabled(true)->setTtl($this->defaultTtl);
            $option = $route->getOption('cache');

            if (is_bool($option)) {
                $cache->setEnabled($option);
            } else if (is_array($option)) {
                if (isset($option['enabled'])) {
                    $cache->setEnabled($option['enabled']);
                }
                if (isset($option['ttl'])) {
                    $cache->setTtl($option['ttl']);
                }
            }

            $route->setDefault('_cache_control', $cache);
        }
    }

    /**
     * @param FilterResponseEvent $event
     */
    public function onKernelResponse(FilterResponseEvent $event)
    {
        $response = $event->getResponse();

        if (!$response->isSuccessful()) {
            return;
        }

        /**
         * If the request has annotated cache directives, skip
         */
        if (class_exists(\Sensio\Bundle\FrameworkExtraBundle\Configuration\Cache::class) &&
            $event->getRequest()->get('_cache') instanceof \Sensio\Bundle\FrameworkExtraBundle\Configuration\Cache) {
            return;
        }

        $cache = $event->getRequest()->get('_cache_control');
        if ($cache instanceof CacheControl) {
            if (!$cache->isEnabled()) {
                return $event->setResponse($this->disableCache($response));
            }
            return $event->setResponse($this->enableCache($response, $cache->getTtl()));
        }
    }

    /**
     * @param Response $response
     * @param int      $ttl
     *
     * @return Response
     */
    protected function enableCache(Response $response, $ttl = null)
    {
        $ttl = $ttl ? $ttl : $this->defaultTtl;

        $response->setPublic()->setMaxAge($ttl);
        $response->setPublic()->setSharedMaxAge($ttl);
        // To be sure we never have "no-store" in headers (for example, when forwarding controller)
        $response->headers->removeCacheControlDirective('no-store');

        return $response;
    }

    /**
     * @param Response $response
     *
     * @return Response
     */
    protected function disableCache(Response $response)
    {
        $response->setPrivate();
        $response->headers->addCacheControlDirective('no-store');

        return $response;
    }
}
