<?php

namespace AlloCine\BrandBundle\Command;

use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Process\Process;
use ModelBundle\Constants\Utils;

/**
 * Description of CacheWarmupupAllCommand
 */
class CacheWarmupupAllCommand extends Base
{
    use ContainerAwareTrait;

    /**
     * Command constructor.
     *
     * @param ParameterBagInterface $parameterBag
     * @param string|null           $name
     */
    public function __construct(ParameterBagInterface $parameterBag, string $name = null) {
        parent::__construct($parameterBag, $name);

        $this->parameterBag = $parameterBag;
    }

    /**
     * @required
     * @param KernelInterface $kernel
     */
    public function setKernel(KernelInterface $kernel): void
    {
        $this->kernel = $kernel;
    }

    /**
     * @required
     * @param ContainerInterface $container
     */
    public function setContainer(ContainerInterface $container = null)
    {
        $this->container = $container;
    }


    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('cache:warmup:all')
            ->setDefinition([
                new InputOption('no-optional-warmers', '', InputOption::VALUE_NONE,
                    'Skip optional cache warmers (faster)'),
            ])
            ->setDescription('Warms up an empty cache fro every brands')
            ->setHelp(<<<'EOF'
The <info>%command.name%</info> command warms up the cache.

Before running this command, the cache must be empty.

This command does not generate the classes cache (as when executing this
command, too many classes that should be part of the cache are already loaded
in memory). Use <comment>curl</comment> or any other similar tool to warm up
the classes cache if you want.

EOF
            );
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $brands = $this->container->get('brand.brand_repository')->getAvailableBrands();
        $consoleExecutable = $this->parameterBag->get('kernel.project_dir') . '/bin/console';

        foreach ($brands as $brand) {
            $io->title(sprintf('Warming up the cache for the <info>%s</info> brand', $brand));

            $command = [
                $consoleExecutable,
                'cache:warmup',
                '--env',
                $this->kernel->getEnvironment(),
                '--brand',
                $brand,
            ];

            if (true === $input->getOption('no-optional-warmers')) {
                $command[] = '--no-optional-warmers';
            }

            if (true === $input->getOption('no-debug')) {
                $command[] = '--no-debug';
            }

            $process = new Process($command);
            $process->setTimeout(null);

            $process->run(function ($type, $text) use ($output) {
                $output->write($text);
            });

            if (!$process->isSuccessful()) {
                throw new \RuntimeException($process->getErrorOutput());
            }
            $process = null;
            $command = null;
        }
    }
}
