<?php

namespace AlloCine\BrandBundle\Kernel;

use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\HttpKernel\Kernel;
use Symfony\Component\Config\Loader\LoaderInterface;

abstract class BrandableKernel extends Kernel
{
    const CONFIG_EXTS = '.{php,xml,yaml,yml}';

    /**
     * @var string
     */
    private $brand;

    /**
     * @param string $environment
     * @param bool $debug
     * @param null $brand
     */
    public function __construct($environment, $debug, $brand = 'allocine')
    {
        $this->brand = $brand;

        parent::__construct($environment, $debug);
    }

    /**
     * sf 3.3 default cache folder
     * @return string
     */
    public function getCacheDir()
    {
        return $this->getProjectDir().'/var/cache/'.$this->getEnvironment().'/'.$this->brand;
    }

    /**
     * sf 3.3 default log folder
     * @return string
     */
    public function getLogDir()
    {
        return $this->getProjectDir().'/var/logs';
    }

    /**
     * sf 3.3 default config folder
     * @return string
     */
    public function getConfigDir()
    {
        return $this->getProjectDir().'/app/config';
    }

    /**
     * @param ContainerBuilder $container
     * @param LoaderInterface  $loader
     *
     * @throws \Exception
     */
    protected function configureContainer(ContainerBuilder $container, LoaderInterface $loader)
    {
        $envconfig = $this->getConfigDir().'/config_'.$this->getEnvironment(). self::CONFIG_EXTS;

        if (!empty(glob($envconfig, GLOB_BRACE))) {
            $loader->load($envconfig, 'glob');
        }

        $brandEnvConfig = $this->getConfigDir().'/'.$this->brand.'/config_'.$this->getEnvironment().self::CONFIG_EXTS;
        $brandConfig = $this->getConfigDir().'/'.$this->brand.'/config' . self::CONFIG_EXTS;

        if (!empty(glob($brandEnvConfig, GLOB_BRACE))) {
            $loader->load($brandEnvConfig, 'glob');
        } elseif (!empty(glob($brandConfig, GLOB_BRACE))) {
            $loader->load($brandConfig, 'glob');
        }
    }

    /**
     * @return string
     */
    public function getBrand()
    {
        return $this->brand;
    }

    /**
     * @param string $brand
     *
     * @throws \LogicException
     */
    public function setBrand($brand)
    {
        if (true === $this->booted) {
            throw new \LogicException('Brand can only be changed when kernel is not booted');
        }

        $this->brand = $brand;
    }

    /**
     * @return string
     */
    public function getName()
    {
        return ucfirst($this->brand) . ($this->isDebug() ? 'Debug' : '' );
    }

    /**
     * @return string
     */
    protected function getContainerClass()
    {
        return $this->name.ucfirst($this->environment).'ProjectContainer';
    }

    /**
     * {@inheritDoc}
     */
    public function serialize()
    {
        return serialize(array($this->environment, $this->debug, $this->brand));
    }

    /**
     * {@inheritDoc}
     */
    public function unserialize($data)
    {
        list($environment, $debug, $brand) = unserialize($data);

        $this->__construct($environment, $debug, $brand);
    }

    /**
     * {@inheritDoc}
     */
    protected function getKernelParameters()
    {
        return array_merge(parent::getKernelParameters(), [
            'brand' => $this->brand
        ]);
    }
}
