<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Tests\Model;

use InvalidArgumentException;
use AlloCine\BrandBundle\Model\FolderRepository;
use PHPUnit\Framework\TestCase;

final class FolderRepositoryTest extends TestCase
{
    public function testGetTaggedFolders(): void
    {
        $folders = $this->getRepository()->getTaggedFolders('twig', 'brand1');
        $this->assertSame(['/view/brand1', '/view/default'], $folders);

        $folders = $this->getRepository()->getTaggedFolders('custom', 'brand2');
        $this->assertSame(['/config/brand2/custom'], $folders);
    }

    public function testGetTaggedFoldersDefault(): void
    {
        $folders = $this->getRepository('brand1')->getTaggedFolders('custom');
        $this->assertSame(['/config/brand1/custom'], $folders);
    }

    public function testGetAllTaggedFolders(): void
    {
        $folders = $this->getRepository()->getAllTaggedFolders('twig');
        $this->assertSame(['/view/brand1', '/view/brand2', '/view/default'], $folders);

        $folders = $this->getRepository()->getAllTaggedFolders('custom');
        $this->assertSame(['/config/brand1/custom', '/config/brand2/custom'], $folders);
    }

    public function testGetAllTaggedFoldersNoBrand(): void
    {
        $this->expectException(InvalidArgumentException::class);
        $this->expectExceptionMessage('No provided brand and no default brand set.');

        $this->getRepository()->getTaggedFolders('twig');
    }

    /**
     * @param string|null $defaultBrand
     */
    private function getRepository($defaultBrand = null): FolderRepository
    {
        $folders = [
            'views' => [
                'path'         => '/view/{brand}',
                'tags'         => ['twig'],
                'commonFolder' => 'default'
            ],
            'custom_config' => [
                'path'         => '/config/{brand}/custom',
                'tags'         => ['custom'],
                'commonFolder' => null,
            ]
        ];

        return new FolderRepository(['brand1', 'brand2'], $folders, $defaultBrand);
    }
}
