<?php

declare(strict_types=1);

namespace AlloCine\BrandBundle\Command;

use RuntimeException;
use AlloCine\BrandBundle\Model\BrandRepository;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\HttpKernel\KernelInterface;
use Symfony\Component\Process\Process;

#[AsCommand(name: 'cache:warmup:all', description: 'Warms up the cache for every brands', help: <<<'TXT'
The <info>%command.name%</info> command warms up the cache.

Before running this command, the cache must be empty.

This command does not generate the classes cache (as when executing this
command, too many classes that should be part of the cache are already loaded
in memory). Use <comment>curl</comment> or any other similar tool to warm up
the classes cache if you want.

TXT)]
class CacheWarmupAllCommand
{
    public function __construct(protected readonly string $projectDir, private readonly BrandRepository $brandRepository, private readonly KernelInterface $kernel)
    {
    }

    protected function configure()
    {
        $this
            ->setDefinition([
                new InputOption(
                    'no-optional-warmers',
                    '',
                    InputOption::VALUE_NONE,
                    'Skip optional cache warmers (faster)'
                ),
            ]);
    }

    public function __invoke(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $brands = $this->brandRepository->getAvailableBrands();
        $consoleExecutable = $this->projectDir . '/bin/console';

        foreach ($brands as $brand) {
            $io->title(sprintf('Warming up the cache for the <info>%s</info> brand', $brand));

            $command = [
                $consoleExecutable,
                'cache:warmup',
                '--env',
                $this->kernel->getEnvironment(),
                '--brand',
                $brand,
            ];

            if (true === $input->getOption('no-optional-warmers')) {
                $command[] = '--no-optional-warmers';
            }

            if (true === $input->getOption('no-debug')) {
                $command[] = '--no-debug';
            }

            $process = new Process($command);
            $process->setTimeout(null);

            $process->run(function ($type, $text) use ($output): void {
                $output->write($text);
            });

            if (!$process->isSuccessful()) {
                throw new RuntimeException($process->getErrorOutput());
            }
            $process = null;
            $command = null;
        }

        return Command::SUCCESS;
    }
}
